<?php
/**
 * Module Soldes Avancées pour PrestaShop 8.0.0+
 * salesmanager/salesmanager.php
 *
 * FICHIER PRINCIPAL - À placer dans /modules/salesmanager/salesmanager.php
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

class SalesManager extends Module
{
    public function __construct()
    {
        $this->name = 'salesmanager';
        $this->tab = 'pricing_promotion';
        $this->version = '1.0.0';
        $this->author = 'SplineOnline';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = ['min' => '8.0.0', 'max' => _PS_VERSION_];
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Gestionnaire de Soldes Avancé');
        $this->description = $this->l('Gérez vos soldes avec démarques, pourcentages et personnalisation visuelle');
        $this->confirmUninstall = $this->l('Êtes-vous sûr de vouloir désinstaller ce module ?');
    }

    public function install()
    {
        if (!parent::install() ||
            !$this->registerHook('displayProductPriceBlock') ||
            !$this->registerHook('displayProductListReviews') ||
            !$this->registerHook('actionProductListModifier') ||
            !$this->registerHook('displayAdminProductsExtra') ||
            !$this->registerHook('actionProductUpdate') ||
            !$this->registerHook('actionProductDelete') ||
            !$this->registerHook('displayShoppingCartProduct')) {
            return false;
        }

        // Créer la table dans la base de données
        $sql = 'CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'product_sales` (
            `id_product` INT(10) UNSIGNED NOT NULL PRIMARY KEY,
            `active` TINYINT(1) UNSIGNED NOT NULL DEFAULT \'0\',
            `sale_percentage` INT(10) UNSIGNED NOT NULL,
            `sale_type` VARCHAR(32) NOT NULL DEFAULT \'other\',
            `badge_bg_color` VARCHAR(7) NOT NULL DEFAULT \'#dc3545\',
            `badge_text_color` VARCHAR(7) NOT NULL DEFAULT \'#ffffff\',
            `price_color` VARCHAR(7) NOT NULL DEFAULT \'#dc3545\',
            `date_from` DATETIME DEFAULT NULL,
            `date_to` DATETIME DEFAULT NULL
        ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8;';

        return Db::getInstance()->execute($sql);
    }

    public function uninstall()
    {
        // Supprimer la table lors de la désinstallation
        $sql = 'DROP TABLE IF EXISTS `'._DB_PREFIX_.'product_sales`';

        return parent::uninstall() && Db::getInstance()->execute($sql);
    }
    
    //---------------------------------------------------------
    //  Hooks du Front Office
    //---------------------------------------------------------

    /**
 * Hook pour modifier l'affichage du prix
 *
 * @param array $params
 * @return string
 */
public function hookDisplayProductPriceBlock($params)
{
    // Récupère l'ID du produit depuis les paramètres
    $id_product = (int)$params['product']['id_product'];
    
    // Récupère les données de solde spécifiques à ce produit
    $saleData = $this->getSaleDataForProduct($id_product);
    
    // Vérifie si la vente est active et si elle n'est pas nulle
    if ($saleData && $this->isSaleActive($saleData)) {
        // Calcule le prix soldé
        $originalPrice = $params['product']['price_without_reduction'];
        $salePercentage = (int)$saleData['sale_percentage'];
        $salePrice = $originalPrice * (1 - ($salePercentage / 100));

        // Assigne les variables au template Smarty
        $this->context->smarty->assign([
            'sale_data' => $saleData,
            'original_price' => $originalPrice,
            'sale_price' => $salePrice,
            'ps_version' => _PS_VERSION_,
        ]);
        
        // Rend le template du prix soldé
        return $this->display(__FILE__, 'product_price.tpl');
    }

    return ''; // Ne retourne rien si le produit n'est pas en solde
}


    /**
 * Modifie la liste des produits dans le panier pour y ajouter les données de solde.
 *
 * @param array $params
 */
public function hookActionProductListModifier($params)
{
    if (!isset($params['products']) || !is_array($params['products'])) {
        return;
    }

    foreach ($params['products'] as &$product) {
        // Récupère les données de solde pour chaque produit du panier
        $saleData = $this->getSaleDataForProduct($product['id_product']);

        if ($saleData && $this->isSaleActive($saleData)) {
            // Ajoute les données de solde au tableau du produit
            $product['sale_data'] = $saleData;
            $product['sale_percentage'] = $saleData['sale_percentage'];
            $product['sale_label'] = $this->getSaleLabel($saleData);
        }
    }
}

    /**
     * Hook pour afficher le badge de solde sur les miniatures de produits
     */
    public function hookDisplayProductListReviews($params)
    {
        $id_product = (int)$params['product']['id_product'];
        $saleData = $this->getSaleDataForProduct($id_product);

        if ($saleData && $this->isSaleActive($saleData)) {
            $this->context->smarty->assign([
                'sale_data' => $saleData,
                'sale_label' => $this->getSaleLabel($saleData)
            ]);
            return $this->fetch('module:salesmanager/views/templates/hook/product_badge.tpl');
        }
    }

    /**
     * Hook pour afficher le prix de solde dans le panier
     */
    public function hookDisplayShoppingCartProduct($params)
    {
        $id_product = (int)$params['product']['id_product'];
        $saleData = $this->getSaleDataForProduct($id_product);
        
        if ($saleData && $this->isSaleActive($saleData)) {
            $originalPrice = Product::getPriceStatic($id_product, true, null, 6, null, false, false, 1, false, null, null, null, $specific_price_output);
            $reduction_factor = (100 - $saleData['sale_percentage']) / 100;
            $salePrice = $originalPrice * $reduction_factor;

            $this->context->smarty->assign([
                'sale_data' => $saleData,
                'original_price' => $originalPrice,
                'sale_price' => $salePrice,
            ]);

            return $this->fetch('module:salesmanager/views/templates/hook/cart_price.tpl');
        }
    }

    
    //---------------------------------------------------------
    //  Fonctions privées pour la gestion des données
    //---------------------------------------------------------
    
    private function getSaleDataForProduct($id_product)
    {
        $sql = 'SELECT * FROM `'._DB_PREFIX_.'product_sales`
                WHERE `id_product` = ' . (int)$id_product;
        
        return Db::getInstance()->getRow($sql);
    }

    private function getSaleTypes()
    {
        return [
            'first_markdown' => $this->l('1ère démarque'),
            'second_markdown' => $this->l('2ème démarque'),
            'third_markdown' => $this->l('3ème démarque'),
            'other' => $this->l('Autre')
        ];
    }

    private function getSaleLabel($saleData)
    {
        $types = $this->getSaleTypes();
        $label = isset($types[$saleData['sale_type']]) ?
                 $types[$saleData['sale_type']] : $this->l('SOLDES');
        return $label . ' -' . $saleData['sale_percentage'] . '%';
    }

    private function isSaleActive($saleData)
    {
        if (!$saleData['active']) {
            return false;
        }

        $now = time();
        
        if ($saleData['date_from'] && strtotime($saleData['date_from']) > $now) {
            return false;
        }
        
        if ($saleData['date_to'] && strtotime($saleData['date_to']) < $now) {
            return false;
        }

        return true;
    }
    
    //---------------------------------------------------------
    //  Hooks du Back Office (pour la gestion dans l'admin)
    //---------------------------------------------------------

    public function hookDisplayAdminProductsExtra($params)
    {
        $id_product = (int)$params['id_product'];
        $saleData = $this->getSaleDataForProduct($id_product);

        if (!$saleData) {
            $saleData = ['active' => 0];
        }

        $this->context->smarty->assign([
            'sale_data' => $saleData,
            'sale_types' => $this->getSaleTypes(),
            'module_name' => $this->name,
            'ps_version' => _PS_VERSION_
        ]);
        
        return $this->fetch('module:salesmanager/views/templates/admin/product_tab.tpl');
    }

    public function hookActionProductUpdate($params)
    {
        $id_product = (int)Tools::getValue('id_product');
        $active = (int)Tools::getValue('sale_active');
        $percentage = (int)Tools::getValue('sale_percentage');
        $sale_type = Tools::getValue('sale_type');
        $badge_bg_color = Tools::getValue('badge_bg_color');
        $badge_text_color = Tools::getValue('badge_text_color');
        $price_color = Tools::getValue('price_color');
        $date_from = Tools::getValue('sale_date_from');
        $date_to = Tools::getValue('sale_date_to');

        $data = [
            'id_product' => $id_product,
            'active' => $active,
            'sale_percentage' => $percentage,
            'sale_type' => $sale_type,
            'badge_bg_color' => $badge_bg_color,
            'badge_text_color' => $badge_text_color,
            'price_color' => $price_color,
            'date_from' => $date_from ? $date_from : null,
            'date_to' => $date_to ? $date_to : null,
        ];
        
        $this->saveSaleData($data);
    }
    
    private function saveSaleData($data)
    {
        $existing = $this->getSaleDataForProduct($data['id_product']);
        
        if ($existing) {
            return Db::getInstance()->update('product_sales', $data, 'id_product = ' . (int)$data['id_product']);
        } else {
            return Db::getInstance()->insert('product_sales', $data);
        }
    }

    public function hookActionProductDelete($params)
    {
        $id_product = (int)$params['id_product'];
        $sql = 'DELETE FROM `'._DB_PREFIX_.'product_sales` WHERE `id_product` = ' . (int)$id_product;
        Db::getInstance()->execute($sql);
    }
}