document.addEventListener('DOMContentLoaded', function() {

    var cookieCategories = {
        'analytics': 'analytics',
        'marketing': 'marketing'
    };
    var consentCookieName = 'myrgpd_consent';
    var scriptsToManage = document.querySelectorAll('script[data-cookie-category]');

    function initializeGtag() {
        if (typeof gtag !== 'undefined') {
            gtag('consent', 'default', {
                'analytics_storage': 'denied',
                'ad_storage': 'denied',
                'ad_user_data': 'denied',
                'ad_personalization': 'denied'
            });
        }
    }

    function updateGtagConsent(consent) {
        if (typeof gtag !== 'undefined') {
            gtag('consent', 'update', {
                'analytics_storage': consent.analytics ? 'granted' : 'denied',
                'ad_storage': consent.marketing ? 'granted' : 'denied',
                'ad_user_data': consent.marketing ? 'granted' : 'denied',
                'ad_personalization': consent.marketing ? 'granted' : 'denied'
            });
        }
    }

    function loadScripts(category) {
        scriptsToManage.forEach(function(script) {
            if (script.getAttribute('data-cookie-category') === category) {
                var originalSrc = script.getAttribute('data-src');
                if (originalSrc) {
                    script.src = originalSrc;
                    script.removeAttribute('data-src');
                }
            }
        });
    }

    function manageConsent(action, preferences = null) {
        var consent = {};

        if (action === 'accept') {
            for (var category in cookieCategories) {
                consent[category] = true;
            }
        } else if (action === 'reject') {
            for (var category in cookieCategories) {
                consent[category] = false;
            }
        } else if (action === 'preferences' && preferences) {
            consent = preferences;
        } else {
            return;
        }

        updateGtagConsent(consent);
        setCookie(consentCookieName, JSON.stringify(consent), 365);
        
        for (var category in consent) {
            if (consent[category]) {
                loadScripts(category);
            }
        }
        
        closeModal();
        closePreferencesModal();
    }

    var existingConsent = getCookie(consentCookieName);
    if (existingConsent) {
        var consent = JSON.parse(existingConsent);
        for (var category in consent) {
            if (consent[category]) {
                loadScripts(category);
            }
        }
    } else {
        initializeGtag();
    }


    function setCookie(name, value, days) {
        var expires = '';
        if (days) {
            var date = new Date();
            date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
            expires = '; expires=' + date.toUTCString();
        }
        document.cookie = name + '=' + (value || '') + expires + '; path=/';
    }

    function getCookie(name) {
        var nameEQ = name + '=';
        var ca = document.cookie.split(';');
        for(var i = 0; i < ca.length; i++) {
            var c = ca[i];
            while (c.charAt(0) == ' ') c = c.substring(1, c.length);
            if (c.indexOf(nameEQ) == 0) return c.substring(nameEQ.length, c.length);
        }
        return null;
    }

    var modal = document.getElementById('myrgpd-modal-container');
    var preferencesModal = document.getElementById('myrgpd-preferences-modal-container');

    function closeModal() {
        if (modal) {
            modal.classList.add('hidden');
        }
    }
    
    function closePreferencesModal() {
        if (preferencesModal) {
            preferencesModal.classList.add('hidden');
        }
    }
    
    function openPreferencesModal() {
        var consent = getCookie(consentCookieName) ? JSON.parse(getCookie(consentCookieName)) : {};
        for (var category in cookieCategories) {
            var input = document.querySelector('input[name="' + category + '"]');
            if (input) {
                input.checked = (typeof consent[category] === 'undefined') ? true : consent[category];
            }
        }

        closeModal();
        if (preferencesModal) {
            preferencesModal.classList.remove('hidden');
        }
    }
    
    var openPreferencesLink = document.getElementById('myrgpd-open-preferences');
    if (openPreferencesLink) {
        openPreferencesLink.addEventListener('click', function(e) {
            e.preventDefault();
            openPreferencesModal();
        });
    }
    
    var rejectBtn = document.getElementById('myrgpd-reject');
    if (rejectBtn) {
        rejectBtn.addEventListener('click', function() {
            manageConsent('reject');
        });
    }

    var acceptBtn = document.getElementById('myrgpd-accept');
    if (acceptBtn) {
        acceptBtn.addEventListener('click', function() {
            manageConsent('accept');
        });
    }
    
    var preferencesBtn = document.getElementById('myrgpd-preferences');
    if (preferencesBtn) {
        preferencesBtn.addEventListener('click', function() {
            openPreferencesModal();
        });
    }

    var closeBtn = document.getElementById('myrgpd-close-btn');
    if (closeBtn) {
        closeBtn.addEventListener('click', function() {
            manageConsent('reject');
        });
    }
    
    var savePreferencesBtn = document.getElementById('myrgpd-save-preferences');
    if (savePreferencesBtn) {
        savePreferencesBtn.addEventListener('click', function() {
            var form = document.getElementById('myrgpd-preferences-form');
            var formData = new FormData(form);
            var preferences = {};
            
            for (var category in cookieCategories) {
                preferences[category] = formData.get(category) === 'on';
            }
            
            manageConsent('preferences', preferences);
        });
    }

    // Affiche le pop-up initial uniquement s'il n'y a pas de cookie de consentement existant
    if (!existingConsent && modal) {
        modal.classList.remove('hidden');
    }
});