<?php
/**
 * My Newsletter Module
 * @author Votre Nom
 * @copyright 2025
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 * @version 1.0.0
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

// Inclure la classe NewsletterCampaign
require_once(dirname(__FILE__) . '/classes/NewsletterCampaign.php');

class MyNewsletterModule extends Module
{
    public function __construct()
    {
        $this->name = 'mynewslettermodule';
        $this->tab = 'emailing';
        $this->version = '1.0.0';
        $this->author = 'SplineOnline';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = ['min' => '8.1.0', 'max' => _PS_VERSION_];
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('My Newsletter Module');
        $this->description = $this->l('Module de gestion de newsletter avec éditeur riche');
        $this->confirmUninstall = $this->l('Êtes-vous sûr de vouloir désinstaller ce module ?');
    }

    public function install()
    {
        if (!parent::install() ||
            !$this->registerHook('displayBackOfficeHeader') ||
            !$this->createTables() ||
            !$this->installTab()) {
            return false;
        }

        return true;
    }

    public function uninstall()
    {
        if (!parent::uninstall() ||
            !$this->removeTables() ||
            !$this->uninstallTab()) {
            return false;
        }

        return true;
    }

    private function createTables()
    {
        $sql = [];

        $sql[] = 'CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'mynewsletter_campaigns` (
            `id_campaign` int(11) NOT NULL AUTO_INCREMENT,
            `subject` varchar(255) NOT NULL,
            `content` text NOT NULL,
            `image_url` varchar(500) DEFAULT NULL,
            `date_add` datetime NOT NULL,
            `date_sent` datetime DEFAULT NULL,
            `status` varchar(20) DEFAULT "draft",
            `recipients_count` int(11) DEFAULT 0,
            PRIMARY KEY (`id_campaign`)
        ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8mb4;';

        $sql[] = 'CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'mynewsletter_logs` (
            `id_log` int(11) NOT NULL AUTO_INCREMENT,
            `id_campaign` int(11) NOT NULL,
            `email` varchar(255) NOT NULL,
            `status` varchar(20) NOT NULL,
            `date_sent` datetime NOT NULL,
            PRIMARY KEY (`id_log`),
            KEY `id_campaign` (`id_campaign`)
        ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8mb4;';

        foreach ($sql as $query) {
            if (!Db::getInstance()->execute($query)) {
                return false;
            }
        }

        return true;
    }

    private function removeTables()
    {
        $sql = [];
        $sql[] = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'mynewsletter_logs`';
        $sql[] = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'mynewsletter_campaigns`';

        foreach ($sql as $query) {
            if (!Db::getInstance()->execute($query)) {
                return false;
            }
        }

        return true;
    }

    private function installTab()
    {
        $tab = new Tab();
        $tab->active = 1;
        $tab->class_name = 'AdminMyNewsletter';
        $tab->name = [];

        foreach (Language::getLanguages(true) as $lang) {
            $tab->name[$lang['id_lang']] = 'My Newsletter';
        }

        $tab->id_parent = (int)Tab::getIdFromClassName('AdminParentModulesSf');
        $tab->module = $this->name;

        return $tab->add();
    }

    private function uninstallTab()
    {
        $id_tab = (int)Tab::getIdFromClassName('AdminMyNewsletter');

        if ($id_tab) {
            $tab = new Tab($id_tab);
            return $tab->delete();
        }

        return true;
    }

    public function getContent()
    {
        Tools::redirectAdmin($this->context->link->getAdminLink('AdminMyNewsletter'));
    }

    public function getNewsletterSubscribers()
    {
        $sql = 'SELECT DISTINCT c.id_customer, c.email, c.firstname, c.lastname
                FROM ' . _DB_PREFIX_ . 'customer c
                WHERE c.newsletter = 1
                AND c.active = 1
                AND c.deleted = 0
                AND c.email != ""
                GROUP BY c.email';

        $customers = Db::getInstance()->executeS($sql);

        if (Db::getInstance()->executeS("SHOW TABLES LIKE '" . _DB_PREFIX_ . "newsletter'")) {
            $sql_newsletter = 'SELECT DISTINCT email, "" as firstname, "" as lastname
                              FROM ' . _DB_PREFIX_ . 'newsletter
                              WHERE active = 1
                              AND email NOT IN (SELECT email FROM ' . _DB_PREFIX_ . 'customer)';

            $newsletter_subs = Db::getInstance()->executeS($sql_newsletter);
            $customers = array_merge($customers, $newsletter_subs);
        }

        return $customers;
    }

    public function sendNewsletter($id_campaign)
    {
        $campaign = new NewsletterCampaign($id_campaign);

        if (!Validate::isLoadedObject($campaign)) {
            return ['success' => false, 'error' => 'Campaign not found'];
        }

        $recipients = $this->getNewsletterSubscribers();

        if (empty($recipients)) {
            return ['success' => false, 'error' => 'Aucun destinataire trouvé'];
        }

        $sent_count = 0;
        $errors = [];

        foreach ($recipients as $recipient) {
            $html_content = $this->buildEmailTemplate($campaign, $recipient);

            $mail_vars = [
                '{subject}' => $campaign->subject,
                '{content}' => $html_content
            ];

            try {
                if (Mail::Send(
                    (int)Configuration::get('PS_LANG_DEFAULT'),
                    'newsletter',
                    $campaign->subject,
                    $mail_vars,
                    $recipient['email'],
                    $recipient['firstname'] . ' ' . $recipient['lastname'],
                    Configuration::get('PS_SHOP_EMAIL'),
                    Configuration::get('PS_SHOP_NAME'),
                    null,
                    null,
                    dirname(__FILE__) . '/mails/',
                    false,
                    (int)Shop::getContextShopID()
                )) {
                    $sent_count++;
                    $this->logEmailSent($id_campaign, $recipient['email'], 'sent');
                } else {
                    $errors[] = $recipient['email'];
                    $this->logEmailSent($id_campaign, $recipient['email'], 'failed');
                }
            } catch (Exception $e) {
                $errors[] = $recipient['email'];
                $this->logEmailSent($id_campaign, $recipient['email'], 'error');
            }
        }

        // Mettre à jour le statut de la campagne
        Db::getInstance()->update('mynewsletter_campaigns', [
            'status' => 'sent',
            'date_sent' => date('Y-m-d H:i:s'),
            'recipients_count' => $sent_count
        ], 'id_campaign = ' . (int)$id_campaign);

        return [
            'success' => true,
            'sent_count' => $sent_count,
            'errors' => $errors
        ];
    }

    private function logEmailSent($id_campaign, $email, $status)
    {
        return Db::getInstance()->insert('mynewsletter_logs', [
            'id_campaign' => (int)$id_campaign,
            'email' => pSQL($email),
            'status' => pSQL($status),
            'date_sent' => date('Y-m-d H:i:s')
        ]);
    }

    public function buildEmailTemplate($campaign, $recipient)
    {
        $shop_name = Configuration::get('PS_SHOP_NAME');
        $shop_url = Context::getContext()->shop->getBaseURL(true);
        $shop_logo = _PS_IMG_ . Configuration::get('PS_LOGO');

        $firstname = !empty($recipient['firstname']) ? $recipient['firstname'] : '';
        $lastname = !empty($recipient['lastname']) ? $recipient['lastname'] : '';
        $email = $recipient['email'];

        $content = $campaign->content;
        $content = str_replace(
            ['{firstname}', '{lastname}', '{email}', '{shop_name}', '{shop_url}'],
            [$firstname, $lastname, $email, $shop_name, $shop_url],
            $content
        );

        if (!empty($campaign->image_url)) {
            $image_html = '<img src="' . $campaign->image_url . '" alt="Newsletter" style="max-width: 100%; height: auto; margin-bottom: 20px;">';
            $content = $image_html . $content;
        }

        $html = '<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>' . htmlspecialchars($campaign->subject) . '</title>
</head>
<body style="margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f4f4f4;">
    <table width="100%" cellpadding="0" cellspacing="0" border="0">
        <tr>
            <td align="center" style="padding: 20px 0;">
                <table width="600" cellpadding="0" cellspacing="0" border="0" style="background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                    <tr>
                        <td style="background-color: #25b9d7; padding: 30px; text-align: center;">
                            <img src="' . $shop_logo . '" alt="' . $shop_name . '" style="max-width: 200px; height: auto;">
                            <h1 style="color: #ffffff; margin: 15px 0 0 0; font-size: 24px;">' . htmlspecialchars($campaign->subject) . '</h1>
                        </td>
                    </tr>

                    <tr>
                        <td style="padding: 30px;">
                            ' . $content . '
                        </td>
                    </tr>

                    <tr>
                        <td style="background-color: #f8f9fa; padding: 20px; text-align: center; font-size: 12px; color: #666666;">
                            <p style="margin: 5px 0;">© ' . date('Y') . ' ' . $shop_name . ' - Tous droits réservés</p>
                            <p style="margin: 5px 0;">
                                <a href="' . $shop_url . '" style="color: #25b9d7; text-decoration: none;">Visiter notre boutique</a>
                            </p>
                            <p style="margin: 5px 0;">
                                Cet email a été envoyé à ' . $email . '<br>
                                <a href="' . Context::getContext()->link->getPageLink('my-account') . '" style="color: #666666; text-decoration: underline;">Gérer mes préférences</a>
                            </p>
                        </td>
                    </tr>
                </table>
            </td>
        </tr>
    </table>
</body>
</html>';

        return $html;
    }

    public function hookDisplayBackOfficeHeader()
    {
        if (Tools::getValue('configure') == $this->name || Tools::getValue('controller') == 'AdminMyNewsletter') {
            $this->context->controller->addJS($this->_path . 'views/js/back.js');
            $this->context->controller->addCSS($this->_path . 'views/css/back.css');
        }
    }
}