<?php
/**
 * AdminMyNewsletterController.php
 * Contrôleur pour l'administration du module newsletter
 */

class AdminMyNewsletterController extends ModuleAdminController
{
    public function __construct()
    {
        $this->bootstrap = true;
        $this->table = 'mynewsletter_campaigns';
        $this->className = 'NewsletterCampaign';
        $this->lang = false;
        $this->identifier = 'id_campaign';
        $this->context = Context::getContext();
        $this->list_no_link = true; // Empêche le lien sur la ligne de la liste

        parent::__construct();

        $this->bulk_actions = [
            'delete' => [
                'text' => $this->l('Supprimer'),
                'icon' => 'icon-trash',
                'confirm' => $this->l('Êtes-vous sûr de vouloir supprimer les éléments sélectionnés ?')
            ]
        ];

        $this->fields_list = [
            'id_campaign' => [
                'title' => $this->l('ID'),
                'width' => 30,
                'align' => 'center'
            ],
            'subject' => [
                'title' => $this->l('Sujet'),
                'width' => 'auto'
            ],
            'status' => [
                'title' => $this->l('Statut'),
                'width' => 100,
                'align' => 'center',
                'callback' => 'getStatusBadge'
            ],
            'recipients_count' => [
                'title' => $this->l('Destinataires'),
                'width' => 100,
                'align' => 'center',
            ],
            'date_add' => [
                'title' => $this->l('Date création'),
                'width' => 150,
                'type' => 'datetime'
            ],
            'date_sent' => [
                'title' => $this->l('Date envoi'),
                'width' => 150,
                'type' => 'datetime'
            ]
        ];

        $this->addRowAction('view');
        $this->addRowAction('edit');
        $this->addRowAction('delete');
        $this->addRowAction('duplicate');
        $this->addRowAction('send');
    }

    public function renderList()
    {
        $this->toolbar_title = $this->l('Gestion des Newsletters');

        $this->toolbar_btn['new'] = [
            'href' => self::$currentIndex . '&add' . $this->table . '&token=' . $this->token,
            'desc' => $this->l('Nouvelle campagne')
        ];

        return parent::renderList();
    }

    public function renderForm()
    {
        $this->fields_form = [
            'tinymce' => true,
            'legend' => [
                'title' => $this->l('Campagne Newsletter'),
                'icon' => 'icon-envelope'
            ],
            'input' => [
                [
                    'type' => 'text',
                    'label' => $this->l('Sujet'),
                    'name' => 'subject',
                    'required' => true,
                    'lang' => false,
                    'col' => 6,
                    'hint' => $this->l('Sujet de l\'email qui sera envoyé')
                ],
                [
                    'type' => 'textarea',
                    'label' => $this->l('Contenu'),
                    'name' => 'content',
                    'required' => true,
                    'autoload_rte' => true,
                    'lang' => false,
                    'cols' => 60,
                    'rows' => 20,
                    'class' => 'rte',
                    'hint' => $this->l('Contenu HTML de la newsletter')
                ],
                [
                    'type' => 'file',
                    'label' => $this->l('Image principale'),
                    'name' => 'image',
                    'display_image' => true,
                    'hint' => $this->l('Image qui sera affichée en haut de la newsletter')
                ],
                [
                    'type' => 'switch',
                    'label' => $this->l('Envoyer immédiatement'),
                    'name' => 'send_now',
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'send_now_on',
                            'value' => 1,
                            'label' => $this->l('Oui')
                        ],
                        [
                            'id' => 'send_now_off',
                            'value' => 0,
                            'label' => $this->l('Non')
                        ]
                    ],
                    'hint' => $this->l('Envoyer la newsletter immédiatement après sauvegarde')
                ],
                [
                    'type' => 'html',
                    'name' => 'preview_button',
                    'html_content' => '<button type="button" class="btn btn-info" id="preview-button">
                        <i class="icon-eye"></i> ' . $this->l('Prévisualiser') . '</button>'
                ]
            ],
            'submit' => [
                'title' => $this->l('Sauvegarder')
            ],
            'buttons' => [
                'test' => [
                    'title' => $this->l('Envoyer un test'),
                    'name' => 'submitTest' . $this->table,
                    'type' => 'submit',
                    'class' => 'btn btn-warning pull-right',
                    'icon' => 'process-icon-mail-send'
                ]
            ]
        ];

        if ($this->object && $this->object->id) {
            $stats = $this->getCampaignStats($this->object->id);

            $this->fields_form['input'][] = [
                'type' => 'html',
                'label' => $this->l('Statistiques'),
                'name' => 'stats',
                'html_content' => $this->renderStats($stats)
            ];
        }

        return parent::renderForm();
    }

    public function postProcess()
    {
        if (Tools::isSubmit('submitTest' . $this->table)) {
            $this->processTestEmail();
            Tools::redirectAdmin(self::$currentIndex . '&' . $this->identifier . '=' . (int)Tools::getValue($this->identifier) . '&update' . $this->table . '&token=' . $this->token);
        }

        if (Tools::isSubmit('send' . $this->table)) {
            $this->processSendCampaign();
        }

        if (Tools::isSubmit('duplicate' . $this->table)) {
            $this->processDuplicate();
        }

        return parent::postProcess();
    }

    public function processSave()
    {
        $id = (int)Tools::getValue('id_campaign');
        if ($id) {
            $campaign = new NewsletterCampaign($id);
        } else {
            $campaign = new NewsletterCampaign();
        }

        $campaign->subject = Tools::getValue('subject');
        $campaign->content = Tools::getValue('content');
        $campaign->status = 'draft';
        $campaign->date_add = date('Y-m-d H:i:s');
        $campaign->date_sent = ''; // Correction : on utilise une chaîne vide pour les brouillons

        // Gérer l'upload d'image
        if (isset($_FILES['image']) && !empty($_FILES['image']['tmp_name'])) {
            $image_upload_path = _PS_MODULE_DIR_ . $this->module->name . '/uploads/';
            if (!is_dir($image_upload_path)) {
                mkdir($image_upload_path, 0777, true);
            }
            $image_name = md5($_FILES['image']['name'] . microtime()) . '.png';
            if (move_uploaded_file($_FILES['image']['tmp_name'], $image_upload_path . $image_name)) {
                $campaign->image_url = $this->module->getModulePathUri() . 'uploads/' . $image_name;
            }
        }

        if ($campaign->add()) {
            if (Tools::isSubmit('send_now')) {
                $this->processSendCampaign();
            }
            $this->redirectAfter = self::$currentIndex . '&conf=3&token=' . $this->token;
            return true;
        } else {
            $this->errors[] = $this->l('Erreur lors de la sauvegarde de la campagne.');
            return false;
        }
    }

    protected function processTestEmail()
    {
        $test_email = Configuration::get('PS_SHOP_EMAIL');
        $subject = Tools::getValue('subject');
        $content = Tools::getValue('content');

        if (empty($subject) || empty($content)) {
            $this->errors[] = $this->l('Le sujet et le contenu sont requis');
            return false;
        }

        $campaign = new NewsletterCampaign();
        $campaign->subject = $subject;
        $campaign->content = $content;

        $recipient = ['email' => $test_email, 'firstname' => '', 'lastname' => ''];
        $html_content = $this->module->buildEmailTemplate($campaign, $recipient);

        if (Mail::Send(
            $this->context->language->id,
            'newsletter',
            '[TEST] ' . $subject,
            ['{subject}' => $subject, '{content}' => $html_content],
            $test_email,
            null,
            Configuration::get('PS_SHOP_EMAIL'),
            Configuration::get('PS_SHOP_NAME'),
            null,
            null,
            _PS_MODULE_DIR_ . $this->module->name . '/mails/',
            false,
            (int)Shop::getContextShopID()
        )) {
            $this->confirmations[] = sprintf($this->l('Email de test envoyé à %s'), $test_email);
        } else {
            $this->errors[] = $this->l('Erreur lors de l\'envoi de l\'email de test');
        }
    }

    protected function processSendCampaign()
    {
        $id_campaign = (int)Tools::getValue('id_campaign');

        if (!$id_campaign) {
            $this->errors[] = $this->l('Campagne invalide');
            return false;
        }

        $result = $this->module->sendNewsletter($id_campaign);

        if (is_array($result)) {
            if (isset($result['success']) && $result['success']) {
                $this->confirmations[] = sprintf(
                    $this->l('Newsletter envoyée à %d destinataires'),
                    isset($result['sent_count']) ? $result['sent_count'] : 0
                );
                if (!empty($result['errors'])) {
                    $this->warnings[] = sprintf(
                        $this->l('Échec pour %d adresses email'),
                        count($result['errors'])
                    );
                }
            } else {
                $error_message = isset($result['error']) ? $result['error'] : $this->l('Erreur inconnue');
                $this->errors[] = $this->l('Erreur lors de l\'envoi: ') . $error_message;
            }
        } else {
            $this->errors[] = $this->l('Réponse invalide du module');
        }

        return true;
    }

    protected function processDuplicate()
    {
        $id = (int)Tools::getValue('id_campaign');
        if (!$id) {
            $this->errors[] = $this->l('Campagne invalide');
            return false;
        }

        $original_campaign = new NewsletterCampaign($id);
        if (!Validate::isLoadedObject($original_campaign)) {
            $this->errors[] = $this->l('Campagne non trouvée');
            return false;
        }

        $new_campaign = clone $original_campaign;
        $new_campaign->id = null;
        $new_campaign->status = 'draft';
        $new_campaign->recipients_count = 0;
        $new_campaign->date_add = date('Y-m-d H:i:s');
        $new_campaign->date_sent = ''; // Correction : on utilise une chaîne vide pour les brouillons

        if ($new_campaign->add()) {
            $this->confirmations[] = $this->l('Campagne dupliquée avec succès.');
            Tools::redirectAdmin(self::$currentIndex . '&token=' . $this->token . '&update' . $this->table . '&id_campaign=' . $new_campaign->id);
        } else {
            $this->errors[] = $this->l('Échec de la duplication.');
        }

        return true;
    }

    public function getStatusBadge($status)
    {
        $badge_class = 'default';
        $label = $status;
        switch ($status) {
            case 'sent':
                $badge_class = 'success';
                $label = $this->l('Envoyée');
                break;
            case 'draft':
                $badge_class = 'warning';
                $label = $this->l('Brouillon');
                break;
        }
        return '<span class="label label-' . $badge_class . '">' . $label . '</span>';
    }

    protected function getCampaignStats($id_campaign)
    {
        $sql = '
            SELECT
                COUNT(*) as total_sent,
                SUM(CASE WHEN status = "sent" THEN 1 ELSE 0 END) as success_count,
                SUM(CASE WHEN status = "failed" THEN 1 ELSE 0 END) as failed_count
            FROM ' . _DB_PREFIX_ . 'mynewsletter_logs
            WHERE id_campaign = ' . (int)$id_campaign;

        return Db::getInstance()->getRow($sql);
    }

    protected function renderStats($stats)
    {
        if (!$stats || !$stats['total_sent']) {
            return '<div class="alert alert-info">' . $this->l('Aucune statistique disponible') . '</div>';
        }
        $success_rate = round(($stats['success_count'] / $stats['total_sent']) * 100, 2);
        $html = '<div class="row">
            <div class="col-md-4"><div class="panel"><div class="panel-heading">
                <i class="icon-envelope"></i> ' . $this->l('Total envoyé') . '</div><div class="panel-body"><h2>' . $stats['total_sent'] . '</h2></div></div></div>
            <div class="col-md-4"><div class="panel"><div class="panel-heading">
                <i class="icon-check"></i> ' . $this->l('Succès') . '</div><div class="panel-body">
                <h2 class="text-success">' . $stats['success_count'] . '</h2><p>' . $success_rate . '%</p></div></div></div>
            <div class="col-md-4"><div class="panel"><div class="panel-heading">
                <i class="icon-warning"></i> ' . $this->l('Échecs') . '</div><div class="panel-body">
                <h2 class="text-danger">' . $stats['failed_count'] . '</h2></div></div></div>
        </div>';
        return $html;
    }

    public function ajaxProcessPreview()
    {
        $subject = Tools::getValue('subject');
        $content = Tools::getValue('content');
        $id_campaign = (int)Tools::getValue('id_campaign');
        $image_url = '';

        if ($id_campaign) {
            $campaign = new NewsletterCampaign($id_campaign);
            $image_url = $campaign->image_url;
        }

        $campaign_data = (object)[
            'subject' => $subject,
            'content' => $this->sanitizeContent($content),
            'image_url' => $image_url,
        ];

        $preview_html = $this->module->buildEmailTemplate(
            $campaign_data,
            (object)['email' => 'test@example.com', 'firstname' => 'John', 'lastname' => 'Doe']
        );

        die($preview_html);
    }

    protected function sanitizeContent($content)
    {
        $allowed_tags = '<p><br><strong><b><em><i><u><span><div><h1><h2><h3><h4><h5><h6><ul><ol><li><a><img><table><tr><td><th><tbody><thead>';
        $content = strip_tags($content, $allowed_tags);
        $content = preg_replace('/on\w+="[^"]*"/i', '', $content);
        $content = preg_replace('/javascript:/i', '', $content);
        return $content;
    }
}