<?php
require_once dirname(__FILE__) . '/../../config/config.inc.php';
require_once dirname(__FILE__) . '/../../init.php';

// HEADERS CORRECTS AVEC ENCODAGE UTF-8
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

$context = Context::getContext();

// Vérification de connexion
$is_logged = $context->customer->isLogged();
if (!$is_logged) {
    $is_logged = $context->employee->isLogged();
}

$action = Tools::getValue('action');

switch ($action) {
    case 'getMessages':
        getMessages($context);
        break;
        
    case 'sendMessage':
        sendMessage($context);
        break;
        
    case 'sendAdminMessage':
        sendAdminMessage($context);
        break;
        
    case 'deleteMessage':
        deleteMessage($context);
        break;
        
    case 'clearHistory':
        clearHistory($context);
        break;
        
    default:
        echo json_encode(['error' => 'Action non reconnue: ' . $action]);
        exit;
}

/**
 * Récupérer les messages
 */
function getMessages($context)
{
    $lastId = (int)Tools::getValue('lastId', 0);
    $limit = 50000;
    
    // Validation stricte des paramètres
    $lastId = (int)Tools::getValue('lastId', 0);
    $limit = 50000; // Hardcodé, pas de user input

    if (!Validate::isUnsignedInt($lastId)) {
        echo json_encode(['error' => 'Invalid parameter']);
        exit;
    }

    $sql = 'SELECT m.*, 
            CASE 
                WHEN m.id_employee > 0 THEN e.firstname
                ELSE c.firstname
            END as firstname,
            CASE 
                WHEN m.id_employee > 0 THEN e.lastname
                ELSE c.lastname
            END as lastname,
            m.id_employee
            FROM `'._DB_PREFIX_.'minichat_messages` m
            LEFT JOIN `'._DB_PREFIX_.'customer` c ON m.id_customer = c.id_customer
            LEFT JOIN `'._DB_PREFIX_.'employee` e ON m.id_employee = e.id_employee
            WHERE m.is_deleted = 0 AND m.id_message > ' . (int)$lastId . '
            ORDER BY m.id_message ASC
            LIMIT ' . (int)$limit;
    
    $messages = Db::getInstance()->executeS($sql);
    
    if (!$messages) {
        $messages = [];
    }
    
    $currentCustomerId = (int)$context->customer->id;
    
    foreach ($messages as &$msg) {
        // Générer les initiales pour l'avatar
        $initials = '';
        $isAdmin = ($msg['id_employee'] > 0);
        
        if (!empty($msg['firstname'])) {
            $initials = strtoupper(substr($msg['firstname'], 0, 1));
            if (!empty($msg['lastname'])) {
                $initials .= strtoupper(substr($msg['lastname'], 0, 1));
            }
        } else {
            $initials = 'G'; // Guest
        }
        
        // Couleurs pour l'avatar
        $colors = [
            ['#667eea', '#764ba2'], // Violet
            ['#f093fb', '#f5576c'], // Rose
            ['#4facfe', '#00f2fe'], // Bleu
            ['#43e97b', '#38f9d7'], // Vert
            ['#fa709a', '#fee140'], // Sunset
            ['#30cfd0', '#330867'], // Ocean
        ];
        
        // Pour les admins, utiliser une couleur spéciale
        if ($isAdmin) {
            $colors = [['#f70c0cff', '#fc9761ff']]; // Rouge/Orange pour admin
            $colorIndex = 0;
        } else {
            $colorIndex = $msg['id_customer'] % count($colors);
        }
        
        // AJOUTER L'AVATAR AU MESSAGE
        $msg['avatar'] = [
            'initials' => $initials,
            'color1' => $colors[$colorIndex][0],
            'color2' => $colors[$colorIndex][1]
        ];
        
        // ID CLIENT FORMATÉ
        if ($isAdmin) {
            $msg['customer_id_display'] = 'ADMIN';
        } else {
            $msg['customer_id_display'] = '#' . str_pad($msg['id_customer'], 4, '0', STR_PAD_LEFT);
        }
        
        // Nom d'affichage
        if ($isAdmin) {
            $msg['display_name'] = 'Admin ' . $msg['firstname'];
        } else {
            $msg['display_name'] = !empty($msg['firstname']) ? $msg['firstname'] : 'Guest';
        }
        
        // Est-ce notre message ?
        $msg['is_own'] = (!$isAdmin && (int)$msg['id_customer'] === $currentCustomerId);
        
        // Heure
        $msg['time'] = date('H:i', strtotime($msg['date_add']));
    }
    
    echo json_encode([
        'success' => true,
        'messages' => $messages
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Envoyer un message (client)
 * 
 */
function sendMessage($context)
{
    if (!$context->customer->isLogged()) {
        echo json_encode(['error' => 'Veuillez vous connecter'], JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    $customerId = (int)$context->customer->id;
    $message = trim(Tools::getValue('message'));
    $maxChars = 200;
    
    if (empty($message)) {
        echo json_encode(['error' => 'Message vide'], JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    // ⭐ SANITISATION RENFORCÉE
    // 1. Strip tags HTML (pas de <script>, <iframe>, etc.)
    $message = strip_tags($message);
    
    // 2. Convertir entités HTML spéciales
    $message = htmlspecialchars($message, ENT_QUOTES, 'UTF-8');
    
    // 3. Limiter longueur
    if (mb_strlen($message, 'UTF-8') > $maxChars) {
        $message = mb_substr($message, 0, $maxChars, 'UTF-8');
    }
    
    // 4. Filtrer caractères dangereux (optionnel mais recommandé)
    $message = preg_replace('/[^\p{L}\p{N}\s\.,!?@#\-]/u', '', $message);
    
    // Insertion sécurisée
    $sql = 'INSERT INTO `'._DB_PREFIX_.'minichat_messages` 
            (`id_customer`, `id_employee`, `message`, `date_add`, `is_deleted`) 
            VALUES (
                '.(int)$customerId.', 
                0, 
                "'.pSQL($message, true).'", 
                "'.date('Y-m-d H:i:s').'",
                0
            )';
    
    if (Db::getInstance()->execute($sql)) {
        echo json_encode(['success' => true], JSON_UNESCAPED_UNICODE);
    } else {
        echo json_encode(['error' => 'Échec envoi'], JSON_UNESCAPED_UNICODE);
    }
    exit;
}

/**
 * Envoyer un message (admin)
 * VERSION ORIGINALE QUI FONCTIONNE
 */
function sendAdminMessage($context)
{
    // Vérifier qu'on est bien appelé depuis le back-office
    $token = Tools::getValue('admin_token');
    $employeeId = (int)Tools::getValue('id_employee');
    
    // Validation simple du token
    if (!$token || !$employeeId) {
        echo json_encode(['error' => 'Non autorisé'], JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    // Vérifier que l'employé existe
    $sql = 'SELECT id_employee FROM `'._DB_PREFIX_.'employee` WHERE id_employee = '.(int)$employeeId.' AND active = 1';
    if (!Db::getInstance()->getValue($sql)) {
        echo json_encode(['error' => 'Employé invalide'], JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    $message = trim(Tools::getValue('message'));
    $maxChars = 200;
    
    if (empty($message)) {
        echo json_encode(['error' => 'Le message ne peut pas être vide'], JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    if (mb_strlen($message, 'UTF-8') > $maxChars) {
        $message = mb_substr($message, 0, $maxChars, 'UTF-8');
    }
    
    $sql = 'INSERT INTO `'._DB_PREFIX_.'minichat_messages` 
            (`id_customer`, `id_employee`, `message`, `date_add`, `is_deleted`) 
            VALUES (
                0, 
                '.(int)$employeeId.', 
                "'.pSQL($message, true).'", 
                "'.date('Y-m-d H:i:s').'",
                0
            )';
    
    if (Db::getInstance()->execute($sql)) {
        echo json_encode(['success' => true], JSON_UNESCAPED_UNICODE);
    } else {
        echo json_encode(['error' => 'Échec de l\'envoi du message'], JSON_UNESCAPED_UNICODE);
    }
    exit;
}

/**
 * Supprimer un message
 */
function deleteMessage($context)
{
    // Vérifier qu'on est bien appelé depuis le back-office
    $token = Tools::getValue('admin_token');
    $employeeId = (int)Tools::getValue('id_employee');
    $messageId = (int)Tools::getValue('message_id');
    
    // Validation
    if (!$token || !$employeeId || !$messageId) {
        echo json_encode(['error' => 'Paramètres invalides ou non autorisé'], JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    // Vérifier que l'employé existe et est actif
    $sql = 'SELECT id_employee FROM `'._DB_PREFIX_.'employee` WHERE id_employee = '.(int)$employeeId.' AND active = 1';
    if (!Db::getInstance()->getValue($sql)) {
        echo json_encode(['error' => 'Employé invalide'], JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    // Soft delete du message (on le marque comme supprimé sans l'effacer)
    $sql = 'UPDATE `'._DB_PREFIX_.'minichat_messages` 
            SET is_deleted = 1 
            WHERE id_message = '.(int)$messageId;
    
    if (Db::getInstance()->execute($sql)) {
        echo json_encode(['success' => true, 'message' => 'Message supprimé avec succès'], JSON_UNESCAPED_UNICODE);
    } else {
        echo json_encode(['error' => 'Échec de la suppression du message'], JSON_UNESCAPED_UNICODE);
    }
    exit;
}

/**
 * Vider l'historique du chat
 */
function clearHistory($context)
{
    // Vérifier que c'est un admin
    if (!$context->employee->isLogged()) {
        echo json_encode(['error' => 'Non autorisé'], JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    $sql = 'UPDATE `'._DB_PREFIX_.'minichat_messages` SET is_deleted = 1';
    
    if (Db::getInstance()->execute($sql)) {
        echo json_encode(['success' => true], JSON_UNESCAPED_UNICODE);
    } else {
        echo json_encode(['error' => 'Échec du vidage'], JSON_UNESCAPED_UNICODE);
    }
    exit;
}