<?php
require_once dirname(__FILE__) . '/../../config/config.inc.php';
require_once dirname(__FILE__) . '/../../init.php';
// Ajoutez cette ligne si vous utilisez des fonctions/classes du module
require_once _PS_MODULE_DIR_ . 'myliveshoppingmodule/myliveshoppingmodule.php'; 

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

$context = Context::getContext();
$db = Db::getInstance();

// ----------------------------------------------------
// VERIFICATION DE L'AUTHENTIFICATION ET DES PERMISSIONS
// ----------------------------------------------------
if (!$context->employee || !$context->employee->isLoggedBack() || !$context->employee->active) {
    http_response_code(403);
    echo json_encode(['error' => 'Accès refusé - Authentification requise']);
    exit;
}

$action = Tools::getValue('action');

// ----------------------------------------------------
// VERIFICATION DU TOKEN (pour les actions sensibles)
// ----------------------------------------------------
if (in_array($action, ['resetLiveStats', 'exportLiveStats']) && 
    (!Tools::getValue('token') || Tools::getValue('token') !== Tools::getAdminTokenLite('AdminModules'))) {
    http_response_code(403);
    echo json_encode(['error' => 'Token de sécurité invalide.'], JSON_UNESCAPED_UNICODE);
    exit;
}
// ----------------------------------------------------

switch ($action) {
    case 'getViewers':
        getViewersCount($db);
        break;
        
    case 'getStock':
        getStockInfo($db);
        break;

    case 'getProductStock':
        getProductStock($db);
        break;
        
    case 'getLiveStats': // NOUVELLE ACTION
        getLiveStats($db);
        break;

    case 'resetLiveStats': // NOUVELLE ACTION
        resetLiveStats();
        break;

    case 'exportLiveStats': // NOUVELLE ACTION
        exportLiveStats($db);
        break;
        
    default:
        echo json_encode(['error' => 'Action inconnue: ' . $action]);
        exit;
}

/**
 * Récupère les statistiques avancées du Live
 */
function getLiveStats($db)
{
    // L'ID du produit actuellement en live (doit être mis à jour par l'admin)
    $live_product_id = (int)Configuration::get('myliveshoppingmodule_ID');
    // L'heure de début du live (mis à jour par resetLiveStats)
    $live_start_time = Configuration::get('MYLIVESHOPPING_LIVE_START_DATE'); 
    
    // Si l'heure de début n'est pas définie, on ne collecte rien
    if (!$live_start_time) {
        echo json_encode([
            'success' => true,
            'stats' => ['live_active' => false, 'message' => 'Live non démarré. Cliquez sur "Démarrer / Réinitialiser le Live" pour commencer la collecte de statistiques.'],
            'timestamp' => date('H:i:s')
        ]);
        exit;
    }
    
    $context = Context::getContext();
    
    try {
        // 1. Nombre de Personnes Connectées (Clientes Uniques)
        // Compte les clients vus sur le live depuis le début (exclut admin 999999 et invités 0)
        $nb_connected_customers = (int)$db->getValue(
            'SELECT COUNT(DISTINCT id_customer) 
             FROM `' . _DB_PREFIX_ . 'live_viewers` 
             WHERE id_customer > 0 AND id_customer != 999999 
             AND last_seen >= "' . pSQL($live_start_time) . '"'
        );

        // 2. Chiffre d'Affaire (Potentiel) et Paniers Actifs
        // Somme des paniers actifs (sans commande associée) créés par ces clients depuis le début du live.
        $stats_carts = $db->getRow('
            SELECT 
                COUNT(DISTINCT c.id_cart) AS nb_active_carts, 
                SUM(c.total_paid) AS ca_potentiel_live
            FROM `' . _DB_PREFIX_ . 'cart` c
            LEFT JOIN `' . _DB_PREFIX_ . 'orders` o ON o.id_cart = c.id_cart
            WHERE c.date_add >= "' . pSQL($live_start_time) . '"
            AND o.id_order IS NULL 
            AND c.id_customer IN (
                SELECT DISTINCT id_customer
                FROM `' . _DB_PREFIX_ . 'live_viewers`
                WHERE id_customer > 0 AND id_customer != 999999 
                AND last_seen >= "' . pSQL($live_start_time) . '"
            )
        ');
        
        $ca_potentiel_live = (float)$stats_carts['ca_potentiel_live'];
        $nb_carts = (int)$stats_carts['nb_active_carts'];

        // 3. Moyenne de Dépense (Potentielle)
        $average_spend = ($nb_carts > 0) ? ($ca_potentiel_live / $nb_carts) : 0;
        
        // 4. Vitesse de Vente (Simplifiée: Temps entre début du live et 1er ajout au panier du produit principal)
        $sale_speed = 'N/A';
        if ($live_product_id) {
            $first_cart_add = $db->getValue('
                SELECT MIN(cp.date_add) 
                FROM `' . _DB_PREFIX_ . 'cart_product` cp
                LEFT JOIN `' . _DB_PREFIX_ . 'cart` c ON c.id_cart = cp.id_cart
                WHERE cp.id_product = ' . (int)$live_product_id . '
                AND c.date_add >= "' . pSQL($live_start_time) . '"
            ');
            
            if ($first_cart_add) {
                $diff = strtotime($first_cart_add) - strtotime($live_start_time);
                $sale_speed = gmdate("H:i:s", $diff); // Format H:M:S
            }
        }

        // ENVOI DE LA RÉPONSE JSON
        echo json_encode([
            'success' => true,
            'stats' => [
                'live_active' => true,
                'live_start_time' => Tools::displayDate($live_start_time, null, true),
                'nb_connected_customers' => $nb_connected_customers,
                'ca_potentiel_live' => Tools::displayPrice($ca_potentiel_live, $context->currency),
                'nb_active_carts' => $nb_carts,
                'average_spend' => Tools::displayPrice($average_spend, $context->currency),
                'sale_speed' => $sale_speed,
                'live_product_id' => $live_product_id,
            ],
            'timestamp' => date('H:i:s')
        ]);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => 'Erreur lors de la récupération des statistiques : ' . $e->getMessage()]);
    }
    exit;
}

/**
 * Réinitialise le temps de début du Live
 */
function resetLiveStats()
{
    // Sécurisé par le token
    $new_start_time = date('Y-m-d H:i:s');
    
    // Réinitialise l'heure de début du live à l'instant présent.
    if (Configuration::updateValue('MYLIVESHOPPING_LIVE_START_DATE', $new_start_time)) {
        // Optionnel: Nettoyer la table live_viewers pour ne laisser que les sessions actives
        // Db::getInstance()->execute('DELETE FROM `' . _DB_PREFIX_ . 'live_viewers`'); 
        
        echo json_encode([
            'success' => true,
            'message' => 'Le Live a été démarré/réinitialisé. Les statistiques sont collectées depuis ' . Tools::displayDate($new_start_time, null, true)
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'error' => 'Échec de la mise à jour de l\'heure de début du Live.'
        ]);
    }
    exit;
}

/**
 * Fonction d'export des statistiques en CSV
 */
function exportLiveStats($db)
{
    // Sécurisé par le token
    $live_product_id = (int)Configuration::get('myliveshoppingmodule_ID');
    $live_start_time = Configuration::get('MYLIVESHOPPING_LIVE_START_DATE'); 
    if (!$live_start_time) {
        die('Le temps de début du live n\'est pas défini pour l\'export.');
    }

    // 1. Récupérer les données pour l'export (clients connectés et leurs paniers actifs)
    $sql = 'SELECT 
                c.id_customer, 
                c.firstname, 
                c.lastname, 
                c.email, 
                MAX(lv.last_seen) as last_seen, 
                MAX(cart.id_cart) as active_cart_id,
                MAX(cart.total_paid) as cart_total
            FROM `' . _DB_PREFIX_ . 'live_viewers` lv
            LEFT JOIN `' . _DB_PREFIX_ . 'customer` c ON c.id_customer = lv.id_customer
            LEFT JOIN `' . _DB_PREFIX_ . 'cart` cart ON cart.id_customer = c.id_customer AND cart.date_add >= "' . pSQL($live_start_time) . '"
            LEFT JOIN `' . _DB_PREFIX_ . 'orders` o ON o.id_cart = cart.id_cart
            WHERE lv.id_customer > 0 AND lv.id_customer != 999999 
            AND lv.last_seen >= "' . pSQL($live_start_time) . '"
            AND o.id_order IS NULL
            GROUP BY c.id_customer
            ORDER BY cart_total DESC';

    $data_to_export = $db->executeS($sql);

    // 2. Préparer les headers CSV
    $filename = 'live_stats_' . date('Ymd_His') . '.csv';
    // Utiliser text/csv pour l'export, c'est le format le plus universel
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    // Ouvrir le flux de sortie
    $output = fopen('php://output', 'w');
    // Définir le séparateur (le point-virgule est souvent préféré en France)
    $separator = ';'; 
    
    // Écrire le BOM (Byte Order Mark) pour assurer le support de l'UTF-8 dans Excel
    fwrite($output, "\xEF\xBB\xBF"); 

    // En-têtes de colonnes
    fputcsv($output, [
        'ID Client', 
        'Prénom', 
        'Nom', 
        'Email', 
        'Dernière Vue', 
        'ID Panier Actif', 
        'Total Panier Potentiel', 
        'ID Produit Live'
    ], $separator);

    // 3. Remplir le CSV
    foreach ($data_to_export as $row) {
        fputcsv($output, [
            $row['id_customer'],
            $row['firstname'],
            $row['lastname'],
            $row['email'],
            $row['last_seen'],
            $row['active_cart_id'] ?? 'N/A',
            $row['cart_total'] ?? '0.00',
            $live_product_id
        ], $separator);
    }

    fclose($output);
    exit;
}