<?php
require_once dirname(__FILE__) . '/../../config/config.inc.php';
require_once dirname(__FILE__) . '/../../init.php';

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

$context = Context::getContext();

// Vérification systématique du token pour TOUTES les requêtes
// Exception : getMessages en lecture seule (pas d'action sensible)
$safe_actions = ['getMessages']; // Actions en lecture seule

$action = Tools::getValue('action');
if ($action && !in_array($action, $safe_actions)) {
    $token = Tools::getValue('token');
    if (!$token || $token !== Tools::getToken(false)) {
        http_response_code(403);
        echo json_encode(['error' => 'Token CSRF invalide', 'code' => 'CSRF_ERROR']);
        exit;
    }
}

// VÉRIFICATION DE CONNEXION SIMPLIFIÉE
$is_logged = $context->customer->isLogged();
if (!$is_logged) {
    $is_logged = $context->employee->isLogged();
}

if (!$is_logged) {
    $data = [
        'product' => null,
        'combinations' => [],
        'message' => 'not_logged_in',
        'login_required' => true,
        'debug' => ['user_logged' => false]
    ];
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

// Vérification du token pour les actions sensibles
if (Tools::getValue('action') && in_array(Tools::getValue('action'), ['sendMessage', 'sendAdminMessage', 'deleteMessage'])) {
    $token = Tools::getValue('token');
    if (!$token || $token !== Tools::getToken(false)) {
        http_response_code(403);
        echo json_encode(['error' => 'Token CSRF invalide']);
        exit;
    }
}

// Récupération des données
$id_product = (int)Configuration::get('myliveshoppingmodule_ID');

// DEBUG COMPLET - Tester le produit
$product_test = new Product($id_product, false, $context->language->id);
$product_exists = Validate::isLoadedObject($product_test);
$product_active = $product_exists ? $product_test->active : false;
$product_name = $product_exists ? $product_test->name : 'N/A';

$debug_info = [
    'user_logged' => true,
    'customer_id' => $context->customer->id,
    'customer_name' => $context->customer->firstname . ' ' . $context->customer->lastname,
    'live_product_id' => $id_product,
    'slide_list' => Configuration::get('myliveshoppingmodule_SLIDE_LIST'),
    'slide_index' => (int)Configuration::get('myliveshoppingmodule_SLIDE_INDEX'),
    'language_id' => $context->language->id,
    'shop_id' => $context->shop->id,
    'product_exists' => $product_exists,
    'product_active' => $product_active,
    'product_name' => $product_name
];

$data = [
    'product' => null,
    'combinations' => [],
    'message' => '',
    'debug' => $debug_info
];

// Vérification activation
if (!(int)Configuration::get('myliveshoppingmodule_ENABLED')) {
    $data['message'] = 'Live shopping désactivé';
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

// Vérification produit
if (!$id_product) {
    $data['message'] = 'Aucun produit sélectionné';
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

// Chargement du produit
$product = new Product($id_product, false, $context->language->id);
if (!Validate::isLoadedObject($product)) {
    $data['message'] = 'Produit introuvable (ID: ' . $id_product . ', Lang: ' . $context->language->id . ')';
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

// Vérifier si le produit est actif
if (!$product->active) {
    $data['message'] = 'Produit désactivé';
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

/**
 * Calcule le stock RÉELLEMENT disponible pour le live
 * = Stock en DB - Produits dans les paniers non validés
 */
function getRealAvailableStock($id_product, $id_product_attribute = 0) {
    $context = Context::getContext();
    $id_shop = (int)$context->shop->id;
    
    // 1️⃣ Lire le stock en base
    $stock_db = (int)Db::getInstance()->getValue(
        'SELECT quantity FROM ' . _DB_PREFIX_ . 'stock_available 
        WHERE id_product = ' . (int)$id_product . ' 
        AND id_product_attribute = ' . (int)$id_product_attribute . ' 
        AND id_shop = ' . (int)$id_shop
    );
    
    // 2️⃣ Compter les produits dans les paniers NON validés (sans commande)
    $sql = 'SELECT SUM(cp.quantity) as reserved
    FROM `' . _DB_PREFIX_ . 'cart_product` cp
    INNER JOIN `' . _DB_PREFIX_ . 'cart` c ON c.id_cart = cp.id_cart
    LEFT JOIN `' . _DB_PREFIX_ . 'orders` o ON o.id_cart = c.id_cart
    WHERE cp.id_product = ' . (int)$id_product . '
    AND cp.id_product_attribute = ' . (int)$id_product_attribute . '
    AND c.id_shop = ' . (int)$id_shop . '
    AND o.id_order IS NULL';

    // Vérification supplémentaire des types AVANT requête
    if (!Validate::isUnsignedId($id_product) || 
        !Validate::isUnsignedId($id_product_attribute) || 
        !Validate::isUnsignedId($id_shop)) {
        PrestaShopLogger::addLog('Invalid ID in getRealAvailableStock', 3);
        return 0;
    }

$reserved = (int)Db::getInstance()->getValue($sql);
    
    // 3️⃣ Stock disponible = Stock DB - Stock réservé dans les paniers
    $available_stock = max(0, $stock_db - $reserved);
    
    PrestaShopLogger::addLog(
        sprintf(
            'Live Stock - Produit=%d, Attr=%d | DB=%d, Réservé=%d, Disponible=%d',
            $id_product,
            $id_product_attribute,
            $stock_db,
            $reserved,
            $available_stock
        ),
        1
    );
    
    return $available_stock;
}

// Image par défaut
$cover = $product->getCover($product->id);
$default_image = '';

// Gestion de l'image par défaut avec vérification
if ($cover && isset($cover['id_image'])) {
    $default_image = $context->link->getImageLink(
        $product->link_rewrite,
        $cover['id_image'],
        'home_default'
    );
} else {
    // Image par défaut si pas de cover
    $default_image = $context->link->getImageLink(
        'default',
        $context->language->iso_code . '-default',
        'home_default'
    );
}

// Stock RÉEL du produit sans déclinaison
$default_stock = getRealAvailableStock($product->id, 0);

// Données produit AVEC STOCK RÉEL
$product_name = $product->name;
if (is_array($product_name)) {
    $product_name = reset($product_name);
}

$data['product'] = [
    'id' => $product->id,
    'name' => $product_name,
    'default_price' => Tools::displayPrice(Product::getPriceStatic($product->id), $context->currency),
    'default_image' => $default_image,
    'link_rewrite' => is_array($product->link_rewrite) ? reset($product->link_rewrite) : $product->link_rewrite,
    'stock' => (int)$default_stock
];

// Combinaisons AVEC STOCK RÉEL
$combinations = $product->getAttributeCombinations($context->language->id);
if ($combinations && is_array($combinations)) {
    foreach ($combinations as $combination) {
        $id_pa = (int)$combination['id_product_attribute'];
        
        // Prix de la combinaison
        $combination_price = Product::getPriceStatic($id_product, true, $id_pa);
        
        // Stock RÉEL pour cette combinaison
        $combination_stock = getRealAvailableStock($id_product, $id_pa);

        // Image de combinaison
        $combination_image = $default_image;
        $attribute_images = $product->getCombinationImages($context->language->id);
        if (isset($attribute_images[$id_pa]) && !empty($attribute_images[$id_pa])) {
            $first_image = reset($attribute_images[$id_pa]);
            if (isset($first_image['id_image'])) {
                $combination_image = $context->link->getImageLink(
                    $product->link_rewrite,
                    $first_image['id_image'],
                    'home_default'
                );
            }
        }

        // Créer ou compléter la combinaison
        if (!isset($data['combinations'][$id_pa])) {
            $data['combinations'][$id_pa] = [
                'id_product_attribute' => $id_pa,
                'price' => Tools::displayPrice($combination_price, $context->currency),
                'image' => $combination_image,
                'stock' => (int)$combination_stock,
                'attributes' => []
            ];
        }

        // Ajouter les attributs
        $data['combinations'][$id_pa]['attributes'][$combination['group_name']] = $combination['attribute_name'];
    }
    
    // Convertir en array simple pour JavaScript
    $data['combinations'] = array_values($data['combinations']);
    
    // Recréer avec les ID comme clés pour compatibilité
    $temp_combinations = [];
    foreach ($data['combinations'] as $comb) {
        $temp_combinations[$comb['id_product_attribute']] = $comb;
    }
    $data['combinations'] = $temp_combinations;
}

// Message de succès
$data['message'] = 'success';

// Envoi de la réponse JSON
echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
exit;