<?php
/**
 * 2025 SplineOnline
 * MyCart - FORCE la synchronisation des paniers clients
 * Version qui FORCE vraiment le chargement des paniers depuis la DB
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

class MyCart extends Module
{
    public function __construct()
    {
        $this->name = 'mycart';
        $this->tab = 'front_office_features';
        $this->version = '1.0.0';
        $this->author = 'SplineOnline';
        $this->need_instance = 0;
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('My Cart - Force Sync & Merge');
        $this->description = $this->l('Permet aux clients de fusionner d\'anciens paniers dans leur panier actuel.');

        $this->ps_versions_compliancy = ['min' => '8.0.0', 'max' => _PS_VERSION_];
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        $hooks = [
            'actionAuthentication',
            'actionCustomerLoginAfter', 
            'actionObjectCustomerLoginAfter',
            'displayTop',
            'actionFrontControllerSetMedia',
            'displayCustomerAccount',
            'displayShoppingCartFooter',
        ];

        foreach ($hooks as $hook) {
            $this->registerHook($hook);
        }

        return true;
    }

    public function uninstall()
    {
        return parent::uninstall();
    }

    public function hookActionAuthentication($params)
    {
        $this->forceCartSync($params);
    }

    public function hookActionCustomerLoginAfter($params)
    {
        $this->forceCartSync($params);
    }

    public function hookActionObjectCustomerLoginAfter($params)
    {
        $adaptedParams = [];
        if (isset($params['object']) && $params['object'] instanceof Customer) {
            $adaptedParams['customer'] = $params['object'];
        }
        $this->forceCartSync($adaptedParams);
    }

    public function hookDisplayTop($params)
    {
        $context = Context::getContext();
        
        if (!$context->customer->isLogged()) {
            return;
        }

        if ((int)$context->cookie->id_cart !== (int)$context->cart->id) {
            $this->ensureCustomerHasCorrectCart();
        }
    }

    public function hookActionFrontControllerSetMedia($params)
    {
        $context = Context::getContext();
        
        if (!$context->customer->isLogged()) {
            return;
        }

        $this->ensureCustomerHasCorrectCart();
    }

    public function hookDisplayCustomerAccount($params)
    {
        $context = Context::getContext();
        
        if (!$context->customer->isLogged()) {
            return;
        }

        $customer = $context->customer;

        $openCarts = $this->getAllOpenCarts($customer->id, (int)$context->cart->id);

        if (empty($openCarts)) {
            return;
        }

        $context->smarty->assign([
            'mycart_open_carts' => $openCarts,
            'mycart_link' => $this->context->link,
        ]);

        return $this->display(__FILE__, 'views/templates/hook/my_account_carts.tpl');
    }

    public function hookDisplayShoppingCartFooter($params)
    {
        $context = Context::getContext();

        if (!$context->customer->isLogged()) {
            return;
        }

        $productsToMerge = $this->getRecoverableProducts((int)$context->customer->id, (int)$context->cart->id);
        
        if (empty($productsToMerge)) {
            return;
        }
        
        $context->smarty->assign([
            'mycart_products_to_merge' => $productsToMerge,
            'mycart_link' => $this->context->link,
            'mycart_customer_id' => (int)$context->customer->id,
        ]);

        return $this->display(__FILE__, 'views/templates/hook/cart_recoverable_products.tpl');
    }

    protected function forceCartSync($params)
    {
        try {
            $customer = $this->extractCustomerFromParams($params);
            if (!$customer) {
                return;
            }

            PrestaShopLogger::addLog(
                '=== MYCART FORCE SYNC START === Customer: ' . $customer->id . ' (' . $customer->firstname . ' ' . $customer->lastname . ')', 
                1
            );

            $this->forceLoadCustomerCartFromDB($customer);

        } catch (Exception $e) {
            PrestaShopLogger::addLog('MyCart forceCartSync error: ' . $e->getMessage(), 3);
        }
    }

    protected function ensureCustomerHasCorrectCart()
    {
        $context = Context::getContext();
        $customer = $context->customer;

        if (!$customer->isLogged()) {
            return;
        }

        $dbCartData = $this->getCustomerCartFromDB($customer->id);
        $currentCart = $context->cart;

        if ($dbCartData && 
            (!$currentCart || 
             (int)$currentCart->id !== (int)$dbCartData['id_cart'] || 
             $currentCart->nbProducts() != $dbCartData['total_products'])) {
            
            PrestaShopLogger::addLog(
                sprintf('MyCart: CORRECTION PANIER - DB Cart %d (%d produits) != Session Cart %d (%d produits)', 
                    $dbCartData['id_cart'], 
                    $dbCartData['total_products'],
                    $currentCart ? $currentCart->id : 0,
                    $currentCart ? $currentCart->nbProducts() : 0
                ), 
                1
            );
            
            $this->forceReplaceCart($dbCartData['id_cart']);
        }
    }

    protected function forceLoadCustomerCartFromDB($customer)
    {
        $context = Context::getContext();
        $currentCart = $context->cart;
        
        $dbCartData = $this->getCustomerCartFromDB($customer->id);
        
        if (!$dbCartData) {
            PrestaShopLogger::addLog('MyCart: Aucun panier trouvé en DB pour client ' . $customer->id, 1);
            return;
        }

        PrestaShopLogger::addLog(
            sprintf('MyCart: Panier DB trouvé - ID:%d, Produits:%d, Client:%s %s', 
                $dbCartData['id_cart'],
                $dbCartData['total_products'],
                $dbCartData['firstname'],
                $dbCartData['lastname']
            ), 
            1
        );

        if ($currentCart && 
            (int)$currentCart->id !== (int)$dbCartData['id_cart'] && 
            $currentCart->nbProducts() > 0) {
            
            PrestaShopLogger::addLog('MyCart: Fusion panier invité -> panier DB', 1);
            $this->mergeGuestCartIntoDB($currentCart, $dbCartData['id_cart']);
        }

        $this->forceReplaceCart($dbCartData['id_cart']);
    }

    protected function getCustomerCartFromDB($id_customer)
    {
        $context = Context::getContext();
        $id_shop = (int)$context->shop->id;

        if (!(int)$id_customer) {
            return null;
        }

        $sql = new DbQuery();
        $sql->select('c.id_cart, c.date_upd, cu.firstname, cu.lastname, cu.email, COUNT(cp.id_product) AS total_products, SUM(cp.quantity) as total_quantity');
        $sql->from('cart', 'c');
        $sql->innerJoin('customer', 'cu', 'cu.id_customer = c.id_customer');
        $sql->leftJoin('orders', 'o', 'o.id_cart = c.id_cart');
        $sql->leftJoin('cart_product', 'cp', 'cp.id_cart = c.id_cart');
        $sql->where('c.id_customer = ' . (int)$id_customer);
        $sql->where('c.id_shop = ' . $id_shop);
        $sql->where('o.id_cart IS NULL');
        $sql->groupBy('c.id_cart');
        $sql->orderBy('c.date_upd DESC');

        $result = Db::getInstance()->getRow($sql);
        
        if ($result && (int)$result['total_products'] > 0) {
            $this->logCartProducts((int)$result['id_cart']);
            return $result;
        }
        
        PrestaShopLogger::addLog('MyCart: AUCUN PANIER avec produits trouvé pour client ' . $id_customer, 1);
        return null;
    }

    protected function getAllOpenCarts($id_customer, $current_cart_id)
    {
        $context = Context::getContext();
        $id_shop = (int)$context->shop->id;

        $sql = new DbQuery();
        $sql->select('c.id_cart, c.date_upd, COUNT(cp.id_product) AS total_products, SUM(cp.quantity) AS total_quantity');
        $sql->from('cart', 'c');
        $sql->innerJoin('cart_product', 'cp', 'cp.id_cart = c.id_cart');
        $sql->leftJoin('orders', 'o', 'o.id_cart = c.id_cart');
        $sql->where('c.id_customer = ' . (int)$id_customer);
        $sql->where('c.id_shop = ' . $id_shop);
        $sql->where('c.id_cart != ' . (int)$current_cart_id);
        $sql->where('o.id_cart IS NULL');
        
        $sql->where('c.date_upd >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)');
        
        $sql->groupBy('c.id_cart');
        $sql->orderBy('c.date_upd DESC');

        $carts = Db::getInstance()->executeS($sql);

        return $carts;
    }

    protected function getRecoverableProducts($id_customer, $current_cart_id)
    {
        $context = Context::getContext();
        $openCarts = $this->getAllOpenCarts($id_customer, $current_cart_id);
        if (empty($openCarts)) {
            return [];
        }

        $cart_ids = array_column($openCarts, 'id_cart');
        $id_lang = (int)$context->language->id;

        $sql = 'SELECT 
                    cp.id_product,
                    cp.id_product_attribute,
                    cp.quantity
                FROM ' . _DB_PREFIX_ . 'cart_product cp
                WHERE cp.id_cart IN (' . implode(',', array_map('intval', $cart_ids)) . ')';

        $products = Db::getInstance()->executeS($sql);
        
        $mergedProducts = [];
        foreach ($products as $product) {
            $key = (int)$product['id_product'] . '-' . (int)$product['id_product_attribute'];
            if (!isset($mergedProducts[$key])) {
                $mergedProducts[$key] = [
                    'id_product' => (int)$product['id_product'],
                    'id_product_attribute' => (int)$product['id_product_attribute'],
                    'quantity' => (int)$product['quantity'],
                ];
            } else {
                $mergedProducts[$key]['quantity'] += (int)$product['quantity'];
            }
        }
        
        $recoverableProducts = [];
        foreach ($mergedProducts as $item) {
            $product_obj = new Product((int)$item['id_product'], false, $id_lang);
            
            if (!Validate::isLoadedObject($product_obj) || !$product_obj->active || !$product_obj->checkAccess($context->customer->id)) {
                continue;
            }

            $product_url = $context->link->getProductLink(
                $product_obj->id,
                $product_obj->link_rewrite,
                null,
                null,
                $id_lang,
                $context->shop->id,
                (int)$item['id_product_attribute']
            );
            
            $image = Image::getCover((int)$item['id_product']);
            $image_link = $context->link->getImageLink($product_obj->link_rewrite, isset($image['id_image']) ? $image['id_image'] : null, 'cart_default');

            $product_price = Product::getPriceStatic((int)$item['id_product'], true, (int)$item['id_product_attribute'], 2, null, false, true);

            $formattedProduct = [
                'id_product' => (int)$item['id_product'],
                'id_product_attribute' => (int)$item['id_product_attribute'],
                'name' => $product_obj->name,
                'price' => Tools::displayPrice($product_price, $context->currency),
                'stock_quantity' => StockAvailable::getQuantityAvailableByProduct((int)$item['id_product'], (int)$item['id_product_attribute']),
                'image_link' => $image_link,
                'url' => $product_url,
                'quantity' => (int)$item['quantity'],
            ];

            // Attributs du produit
            if ((int)$item['id_product_attribute'] > 0) {
                $combination = new Combination((int)$item['id_product_attribute']);
                $formattedProduct['attributes_small'] = $combination->getAttributesName($id_lang);
            } else {
                $formattedProduct['attributes_small'] = '';
            }

            $recoverableProducts[] = $formattedProduct;
        }

        return $recoverableProducts;
    }
    
    public function mergeAllOpenCarts($id_customer, $destination_cart_id)
    {
        $context = Context::getContext();
        $openCarts = $this->getAllOpenCarts($id_customer, $destination_cart_id);
        if (empty($openCarts)) {
            return false;
        }

        $destinationCart = new Cart((int)$destination_cart_id);
        if (!Validate::isLoadedObject($destinationCart)) {
            return false;
        }
        
        foreach ($openCarts as $cartData) {
            $source_cart_id = $cartData['id_cart'];
            $sourceCart = new Cart((int)$source_cart_id);
            if (!Validate::isLoadedObject($sourceCart)) {
                continue;
            }
            
            $products = $sourceCart->getProducts();
            $merge_successful = true;

            foreach ($products as $product) {
                $id_product = (int)$product['id_product'];
                $id_product_attribute = (int)$product['id_product_attribute'];
                $quantity = (int)$product['cart_quantity'];

                $update_success = $destinationCart->updateQty(
                    $quantity, 
                    $id_product, 
                    $id_product_attribute, 
                    null, 
                    'up', 
                    0, 
                    $context->shop, 
                    false
                );
                
                if (!$update_success) {
                    PrestaShopLogger::addLog(
                        sprintf('MyCart: ERREUR lors de la fusion du produit %d (attr:%d) qty:%d du panier %d.', 
                            $id_product, 
                            $id_product_attribute, 
                            $quantity,
                            $source_cart_id
                        ), 
                        3
                    );
                    $merge_successful = false;
                } else {
                     PrestaShopLogger::addLog(
                        sprintf('MyCart: Fusion réussie du produit %d (attr:%d) qty:%d depuis le panier %d.', 
                            $id_product, 
                            $id_product_attribute, 
                            $quantity,
                            $source_cart_id
                        ), 
                        1
                    );
                }
            }
            
            // On ne supprime l'ancien panier que si la fusion a réussi
            if ($merge_successful) {
                $this->deleteCart($sourceCart);
            }
        }

        $destinationCart->update();
        return true;
    }

    protected function logCartProducts($id_cart)
    {
        $id_lang = (int)Context::getContext()->language->id; 
        $sql = 'SELECT 
                    cp.id_product,
                    cp.id_product_attribute,
                    cp.quantity,
                    pl.name as product_name
                FROM ' . _DB_PREFIX_ . 'cart_product cp
                LEFT JOIN ' . _DB_PREFIX_ . 'product_lang pl ON (cp.id_product = pl.id_product AND pl.id_lang = ' . $id_lang . ')
                WHERE cp.id_cart = ' . (int)$id_cart;

        $products = Db::getInstance()->executeS($sql);
        
        if ($products) {
            foreach ($products as $product) {
                PrestaShopLogger::addLog(
                    sprintf('MyCart: Produit en DB - ID:%d, Attr:%d, Qty:%d, Nom:%s', 
                        $product['id_product'],
                        $product['id_product_attribute'],
                        $product['quantity'],
                        $product['product_name']
                    ), 
                    1
                );
            }
        }
    }

    public function forceReplaceCart($id_cart)
    {
        $cart = new Cart((int)$id_cart);
        
        if (!Validate::isLoadedObject($cart)) {
            PrestaShopLogger::addLog('MyCart: ERREUR - Impossible de charger Cart ' . $id_cart, 3);
            return;
        }

        $context = Context::getContext();

        // IMPORTANT: Définir le panier AVANT toute opération
        $context->cart = $cart;
        $context->cookie->id_cart = (int)$cart->id;
        
        // S'assurer que le contexte est complet pour éviter les erreurs
        if (!isset($context->customer) || !$context->customer->id) {
            $context->customer = new Customer((int)$cart->id_customer);
        }
        
        // Écrire le cookie AVANT l'update pour que tous les hooks aient le bon contexte
        $context->cookie->write();

        if (session_status() === PHP_SESSION_ACTIVE) {
            $_SESSION['cart_id'] = (int)$cart->id;
        }

        PrestaShopLogger::addLog(
            sprintf('MyCart: === PANIER FORCE === Cart %d chargé avec %d produits', 
                $cart->id, 
                $cart->nbProducts()
            ), 
            1
        );

        $this->verifyCartIsLoaded($cart);
    }
    
    protected function verifyCartIsLoaded($expectedCart)
    {
        $context = Context::getContext();
        $loadedCart = $context->cart;

        if (!$loadedCart || (int)$loadedCart->id !== (int)$expectedCart->id) {
            PrestaShopLogger::addLog('MyCart: ERREUR - Le panier n\'a pas été correctement chargé!', 3);
        } else {
            PrestaShopLogger::addLog(
                sprintf('MyCart: VERIFICATION OK - Cart %d bien chargé avec %d produits', 
                    $loadedCart->id,
                    $loadedCart->nbProducts()
                ), 
                1
            );
        }
    }

    protected function mergeGuestCartIntoDB($guestCart, $dbCartId)
    {
        $dbCart = new Cart($dbCartId);
        $context = Context::getContext();
        
        if (!Validate::isLoadedObject($dbCart)) {
            return;
        }

        $products = $guestCart->getProducts();
        
        foreach ($products as $product) {
            $id_product = (int)$product['id_product'];
            $id_product_attribute = (int)$product['id_product_attribute'];
            $quantity = (int)$product['cart_quantity'];

            $dbCart->updateQty(
                $quantity, 
                $id_product, 
                $id_product_attribute, 
                null, 
                'up', 
                0, 
                $context->shop, 
                false, 
                true
            );
            
            PrestaShopLogger::addLog(
                sprintf('MyCart: Fusion produit %d (attr:%d) qty:%d dans Cart %d', 
                    $id_product, 
                    $id_product_attribute, 
                    $quantity,
                    $dbCartId
                ), 
                1
            );
        }

        $dbCart->update();

        $this->deleteCart($guestCart);
    }

    protected function deleteCart($cart)
    {
        if (!Validate::isLoadedObject($cart) || !(int)$cart->id) {
            return;
        }

        try {
            Db::getInstance()->delete('cart_product', 'id_cart = ' . (int)$cart->id);
            Db::getInstance()->delete('cart_rule', 'id_cart = ' . (int)$cart->id);
            $cart->delete();
            
            PrestaShopLogger::addLog('MyCart: Panier supprimé - ID ' . $cart->id, 1);
        } catch (Exception $e) {
            PrestaShopLogger::addLog('MyCart: Erreur suppression panier ' . $e->getMessage(), 3);
        }
    }

    protected function extractCustomerFromParams($params)
    {
        if (isset($params['customer']) && $params['customer'] instanceof Customer) {
            return $params['customer'];
        } elseif (isset($params['object']) && $params['object'] instanceof Customer) {
            return $params['object'];
        }
        return null;
    }
}

return true;