/* Mini Chat JavaScript - views/js/minichat.js */

(function() {
    'use strict';
    
    // Wait for MiniChat object to be defined
    function initMiniChat() {
        if (typeof MiniChat === 'undefined') {
            setTimeout(initMiniChat, 100);
            return;
        }
        
        // Extend MiniChat object with methods
        MiniChat.init = function() {
            if (!this.url) return;
            
            // Set up event listeners
            var input = document.getElementById('chat-input');
            if (input) {
                input.addEventListener('input', function() {
                    var counter = document.getElementById('char-count');
                    if (counter) {
                        counter.textContent = this.value.length;
                    }
                });
            }
            
            // Initially hide widget
            var widget = document.getElementById('minichat-widget');
            if (widget) {
                widget.style.display = 'none';
            }
            
            // Start loading messages if logged in
            if (this.isLogged && !this.isBanned) {
                this.loadMessages();
            }
        };
        
        MiniChat.open = function() {
            var widget = document.getElementById('minichat-widget');
            var launcher = document.getElementById('minichat-launcher');
            
            if (widget && launcher) {
                widget.style.display = 'flex';
                launcher.style.display = 'none';
                this.isOpen = true;
                this.unreadCount = 0;
                
                var badge = document.getElementById('unread-badge');
                if (badge) {
                    badge.style.display = 'none';
                }
                
                if (this.isLogged && !this.isBanned) {
                    this.startPolling();
                }
            }
        };
        
        MiniChat.close = function() {
            var widget = document.getElementById('minichat-widget');
            var launcher = document.getElementById('minichat-launcher');
            
            if (widget && launcher) {
                widget.style.display = 'none';
                launcher.style.display = 'flex';
                this.isOpen = false;
                this.stopPolling();
            }
        };
        
        MiniChat.toggle = function() {
            if (this.isOpen) {
                this.close();
            } else {
                this.open();
            }
        };
        
        MiniChat.loadMessages = function() {
            var self = this;
            
            fetch(this.url + '?action=getMessages&lastId=' + this.lastMessageId)
                .then(function(response) {
                    return response.json();
                })
                .then(function(data) {
                    if (data.success && data.messages) {
                        self.displayMessages(data.messages);
                    }
                })
                .catch(function(error) {
                    console.error('Error loading messages:', error);
                });
        };
        
        MiniChat.displayMessages = function(messages) {
            var container = document.getElementById('chat-messages');
            if (!container) return;
            
            // Remove loading message
            var loading = container.querySelector('.minichat-loading');
            if (loading) {
                loading.remove();
            }
            
            var self = this;
            messages.forEach(function(msg) {
                // Check if message already exists
                if (document.querySelector('[data-msg-id="' + msg.id_message + '"]')) {
                    return;
                }
                
                var messageDiv = document.createElement('div');
                messageDiv.className = 'minichat-message ' + (msg.is_own ? 'own' : 'other');
                messageDiv.setAttribute('data-msg-id', msg.id_message);
                
                var html = '';
                if (!msg.is_own) {
                    html += '<div class="name">' + self.escapeHtml(msg.display_name) + '</div>';
                }
                html += '<div class="bubble">' + self.escapeHtml(msg.message) + '</div>';
                html += '<div class="time">' + msg.time + '</div>';
                
                messageDiv.innerHTML = html;
                container.appendChild(messageDiv);
                
                // Update last message ID
                self.lastMessageId = Math.max(self.lastMessageId, parseInt(msg.id_message));
                
                // Update unread count if window not open
                if (!self.isOpen) {
                    self.unreadCount++;
                    var badge = document.getElementById('unread-badge');
                    if (badge) {
                        badge.textContent = self.unreadCount;
                        badge.style.display = 'inline-block';
                    }
                }
            });
            
            // Scroll to bottom
            container.scrollTop = container.scrollHeight;
        };
        
        MiniChat.send = function() {
            var input = document.getElementById('chat-input');
            if (!input || !input.value.trim()) return;
            
            var message = input.value.trim();
            var self = this;
            
            var formData = new FormData();
            formData.append('action', 'sendMessage');
            formData.append('message', message);
            
            fetch(this.url, {
                method: 'POST',
                body: formData
            })
            .then(function(response) {
                return response.json();
            })
            .then(function(data) {
                if (data.success) {
                    input.value = '';
                    var counter = document.getElementById('char-count');
                    if (counter) {
                        counter.textContent = '0';
                    }
                    self.loadMessages();
                } else if (data.error) {
                    alert(data.error);
                }
            })
            .catch(function(error) {
                console.error('Error sending message:', error);
            });
        };
        
        MiniChat.startPolling = function() {
            if (this.timer) return;
            
            var self = this;
            this.timer = setInterval(function() {
                self.loadMessages();
            }, this.refresh);
        };
        
        MiniChat.stopPolling = function() {
            if (this.timer) {
                clearInterval(this.timer);
                this.timer = null;
            }
        };
        
        MiniChat.escapeHtml = function(text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        };
        
        // Initialize
        MiniChat.init();
    }
    
    // Start initialization when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initMiniChat);
    } else {
        initMiniChat();
    }
})();