<?php
/**
 * Module MiniChat pour PrestaShop 1.7/8
 * Version avec support administrateur, export et reset
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

class MiniChat extends Module
{
    public function __construct()
    {
        $this->name = 'minichat';
        $this->tab = 'front_office_features';
        $this->version = '1.0.0';
        $this->author = 'SplineOnline';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = [
            'min' => '1.7.0.0',
            'max' => _PS_VERSION_
        ];
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Mini Chat');
        $this->description = $this->l('Simple chat module for customer support with admin access');
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall?');
    }

    /**
     * Installation du module
     */
    public function install()
    {
        // Installation de base + base de données + hooks
        if (!parent::install() || 
            !$this->installDb() || 
            !$this->registerHook('displayHeader') || 
            !$this->registerHook('displayFooter') ||
            !$this->registerHook('displayBackOfficeHeader')) {
            return false;
        }

        // Configuration par défaut
        Configuration::updateValue('MINICHAT_ENABLED', 1);
        Configuration::updateValue('MINICHAT_MAX_CHARS', 200);
        Configuration::updateValue('MINICHAT_REFRESH_INTERVAL', 3000);
        Configuration::updateValue('MINICHAT_MAX_MESSAGES', 50);

        return true;
    }

    /**
     * Désinstallation du module
     */
    public function uninstall()
    {
        // Suppression de la configuration
        Configuration::deleteByName('MINICHAT_ENABLED');
        Configuration::deleteByName('MINICHAT_MAX_CHARS');
        Configuration::deleteByName('MINICHAT_REFRESH_INTERVAL');
        Configuration::deleteByName('MINICHAT_MAX_MESSAGES');

        // Désinstallation de base + base de données
        if (!parent::uninstall() || !$this->uninstallDb()) {
            return false;
        }

        return true;
    }

    /**
     * Création des tables en base de données
     */
    private function installDb()
    {
        $sql = [];
        
        // Table des messages - MODIFIÉE pour inclure id_employee
        $sql[] = 'CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'minichat_messages` (
            `id_message` int(11) unsigned NOT NULL AUTO_INCREMENT,
            `id_customer` int(11) unsigned NOT NULL DEFAULT 0,
            `id_employee` int(11) unsigned NOT NULL DEFAULT 0,
            `message` text NOT NULL,
            `date_add` datetime NOT NULL,
            `is_deleted` tinyint(1) unsigned NOT NULL DEFAULT 0,
            PRIMARY KEY (`id_message`),
            KEY `id_customer` (`id_customer`),
            KEY `id_employee` (`id_employee`),
            KEY `date_add` (`date_add`)
        ) ENGINE='._MYSQL_ENGINE_.' DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;';

        // Table de modération
        $sql[] = 'CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'minichat_moderation` (
            `id_customer` int(11) unsigned NOT NULL,
            `is_banned` tinyint(1) unsigned NOT NULL DEFAULT 0,
            `ban_reason` varchar(255) DEFAULT NULL,
            `date_add` datetime NOT NULL,
            `date_upd` datetime DEFAULT NULL,
            PRIMARY KEY (`id_customer`)
        ) ENGINE='._MYSQL_ENGINE_.' DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;';

        // Exécution des requêtes
        foreach ($sql as $query) {
            if (!Db::getInstance()->execute($query)) {
                return false;
            }
        }
        
        // Ajouter la colonne id_employee si elle n'existe pas (pour les mises à jour)
        $this->addEmployeeColumnIfNotExists();

        return true;
    }
    
    /**
     * Ajouter la colonne id_employee si elle n'existe pas
     */
    private function addEmployeeColumnIfNotExists()
    {
        $sql = "SHOW COLUMNS FROM `"._DB_PREFIX_."minichat_messages` LIKE 'id_employee'";
        $result = Db::getInstance()->executeS($sql);
        
        if (empty($result)) {
            $sql = "ALTER TABLE `"._DB_PREFIX_."minichat_messages` 
                    ADD `id_employee` int(11) unsigned NOT NULL DEFAULT 0 AFTER `id_customer`,
                    ADD KEY `id_employee` (`id_employee`)";
            Db::getInstance()->execute($sql);
        }
    }

    /**
     * Suppression des tables de la base de données
     */
    private function uninstallDb()
    {
        $sql = [];
        $sql[] = 'DROP TABLE IF EXISTS `'._DB_PREFIX_.'minichat_messages`';
        $sql[] = 'DROP TABLE IF EXISTS `'._DB_PREFIX_.'minichat_moderation`';
        
        foreach ($sql as $query) {
            if (!Db::getInstance()->execute($query)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Configuration du module dans le back-office
     */
    public function getContent()
    {
        $output = '';
        
        // Traitement des actions d'export et de reset
        if (Tools::isSubmit('exportMiniChat')) {
            $this->exportChatToTxt();
        }
        
        if (Tools::isSubmit('resetMiniChat')) {
            if ($this->resetChat()) {
                $output .= $this->displayConfirmation($this->l('Chat history has been cleared successfully'));
            } else {
                $output .= $this->displayError($this->l('Error clearing chat history'));
            }
        }
        
        // Ajouter le widget de chat admin ici
        $output .= $this->displayAdminChat();
        
        // Traitement du formulaire
        if (Tools::isSubmit('submitMiniChat')) {
            $enabled = (int)Tools::getValue('MINICHAT_ENABLED');
            $maxChars = (int)Tools::getValue('MINICHAT_MAX_CHARS');
            $refresh = (int)Tools::getValue('MINICHAT_REFRESH_INTERVAL');
            $maxMessages = (int)Tools::getValue('MINICHAT_MAX_MESSAGES');
            
            // Validation
            $errors = [];
            if ($maxChars < 50 || $maxChars > 500) {
                $errors[] = $this->l('Max characters must be between 50 and 500');
            }
            if ($refresh < 1000 || $refresh > 60000) {
                $errors[] = $this->l('Refresh interval must be between 1000 and 60000 ms');
            }
            if ($maxMessages < 10 || $maxMessages > 5000) {
                $errors[] = $this->l('Max messages must be between 10 and 5000');
            }
            
            if (empty($errors)) {
                Configuration::updateValue('MINICHAT_ENABLED', $enabled);
                Configuration::updateValue('MINICHAT_MAX_CHARS', $maxChars);
                Configuration::updateValue('MINICHAT_REFRESH_INTERVAL', $refresh);
                Configuration::updateValue('MINICHAT_MAX_MESSAGES', $maxMessages);
                $output .= $this->displayConfirmation($this->l('Settings updated successfully'));
            } else {
                foreach ($errors as $error) {
                    $output .= $this->displayError($error);
                }
            }
        }
        
        return $output . $this->renderForm();
    }
    
    /**
     * Afficher le chat pour l'admin dans la page de configuration
     */
    private function displayAdminChat()
    {
        $employee = $this->context->employee;
        if (!$employee || !$employee->id) {
            return '';
        }
        

        // Initialiser la variable output
        $output = '';

        // --- Bloc d'infos avec logo ---
        $output .= '<div class="panel" style="margin-bottom: 20px;">';
        $output .= '<div class="form-group text-center">';
        $output .= '<img src="../modules/'.$this->name.'/logo.png" alt="Logo MiniChat" style="max-width:120px; margin-bottom:15px;">';
        $output .= '<h3 style="color: #333; margin-bottom: 15px;">'.$this->l('Module MiniChat - Chat en temps réel').'</h3>';
        $output .= '<p style="max-width: 800px; margin: 0 auto; line-height: 1.6; color: #666;">';
        $output .= $this->l('Merci d\'utiliser le module MiniChat. Ce tchat permet d\'afficher une bulle de discussion entre les utilisateurs connectés et vous. ');
        $output .= $this->l('Vous pourrez leur répondre depuis ce panneau d\'administration uniquement et gérer vos tchats avec la possibilité de les exporter et nettoyer. ');
        $output .= $this->l('Sur le site ne sera affiché que le prénom de l\'utilisateur et son ID. Bon tchat avec MiniChat !');
        $output .= '</p>';
        $output .= '</div>';
        $output .= '</div>';
        // --- Fin bloc logo ---

        // Compter les messages
        $sql = 'SELECT COUNT(*) FROM `'._DB_PREFIX_.'minichat_messages` WHERE is_deleted = 0';
        $messageCount = Db::getInstance()->getValue($sql);
        
        $html = '
        <div class="panel">
            <h3><i class="icon-comments"></i> ' . $this->l('Chat Support - Admin View') . '</h3>
            
            <div class="row" style="margin-bottom: 15px;">
                <div class="col-md-12">
                    <div class="btn-group">
                        <form method="post" style="display: inline-block; margin-right: 10px;">
                            <button type="submit" name="exportMiniChat" class="btn btn-default">
                                <i class="icon-download"></i> ' . $this->l('Export to TXT') . '
                            </button>
                        </form>
                        <form method="post" style="display: inline-block;" onsubmit="return confirm(\'' . $this->l('Are you sure you want to clear all chat history? This action cannot be undone.') . '\');">
                            <button type="submit" name="resetMiniChat" class="btn btn-danger">
                                <i class="icon-trash"></i> ' . $this->l('Clear Chat History') . '
                            </button>
                        </form>
                        <span class="badge" style="margin-left: 15px; padding: 7px 12px; background: #25b9d7; color: white; font-size: 14px;">
                            ' . sprintf($this->l('%d messages'), $messageCount) . '
                        </span>
                    </div>
                </div>
            </div>
            
            <div id="admin-minichat-container" style="height: 400px; border: 1px solid #ddd; border-radius: 4px; display: flex; flex-direction: column;">
                <div id="admin-chat-messages" style="flex: 1; overflow-y: auto; padding: 15px 35px; background: #f8f9fa; position: relative;"></div>
                <div style="padding: 10px; background: white; border-top: 1px solid #ddd;">
                    <div style="display: flex;">
                        <input type="text" id="admin-chat-input" placeholder="' . $this->l('Type your message...') . '" style="flex: 1; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" maxlength="' . Configuration::get('MINICHAT_MAX_CHARS', 200) . '">
                        <button onclick="AdminMiniChat.send()" class="btn btn-primary" style="margin-left: 10px;">' . $this->l('Send') . '</button>
                    </div>
                </div>
            </div>
            <div style="margin-top: 10px; padding: 10px; background: #f8f9fa; border-radius: 4px;">
                <small style="color: #666;">
                    <i class="icon-info-circle"></i> ' . $this->l('Tip: Hover over messages to see the delete button (×). Deleted messages are hidden but can be recovered from the database if needed.') . '
                </small>
            </div>
        </div>
        
        <script>
        var AdminMiniChat = {
            url: "' . $this->context->link->getModuleLink('minichat', 'api') . '",
            employeeId: ' . (int)$employee->id . ',
            adminToken: "' . Tools::substr(Tools::encrypt('minichat'.$employee->id), 0, 10) . '",
            lastMessageId: 0,
            timer: null,
            
            init: function() {
                this.loadMessages();
                this.startPolling();
                
                document.getElementById("admin-chat-input").addEventListener("keypress", function(e) {
                    if (e.key === "Enter") {
                        AdminMiniChat.send();
                    }
                });
            },
            
            loadMessages: function() {
                fetch(this.url + "?action=getMessages&lastId=" + this.lastMessageId)
                    .then(response => response.json())
                    .then(data => {
                        if (data.success && data.messages) {
                            this.displayMessages(data.messages);
                        }
                    });
            },
            
            displayMessages: function(messages) {
                var container = document.getElementById("admin-chat-messages");
                
                messages.forEach(function(msg) {
                    if (document.querySelector("[data-admin-msg-id=\"" + msg.id_message + "\"]")) {
                        return;
                    }
                    
                    var messageDiv = document.createElement("div");
                    messageDiv.setAttribute("data-admin-msg-id", msg.id_message);
                    messageDiv.style.marginBottom = "10px";
                    messageDiv.style.position = "relative";
                    
                    var isOwnMessage = (msg.id_employee == AdminMiniChat.employeeId);
                    
                    // Bouton de suppression
                    var deleteBtn = "<button onclick=\"AdminMiniChat.deleteMessage(" + msg.id_message + ")\" " +
                        "style=\"position: absolute; top: 0; " + (isOwnMessage ? "left" : "right") + ": -30px; " +
                        "background: #dc3545; color: white; border: none; " +
                        "width: 24px; height: 24px; border-radius: 50%; " +
                        "cursor: pointer; font-size: 12px; opacity: 0.6; transition: opacity 0.2s;\" " +
                        "onmouseover=\"this.style.opacity=1\" onmouseout=\"this.style.opacity=0.6\" " +
                        "title=\"Delete message\">×</button>";
                    
                    messageDiv.innerHTML = 
                        "<div style=\"display: flex; align-items: flex-start; " + (isOwnMessage ? "flex-direction: row-reverse;" : "") + "\">" +
                        deleteBtn +
                        "<div style=\"max-width: 70%;\">" +
                        "<div style=\"font-size: 12px; color: #666; margin-bottom: 4px;\">" +
                        "<strong>" + msg.display_name + "</strong> " +
                        "<span style=\"background: #e9ecef; padding: 2px 6px; border-radius: 10px; font-size: 10px;\">" + msg.customer_id_display + "</span> " +
                        "<span style=\"color: #999;\">" + msg.time + "</span>" +
                        "</div>" +
                        "<div style=\"padding: 10px; border-radius: 10px; background: " + (isOwnMessage ? "#007bff; color: white;" : "#f1f1f1;") + "\">" +
                        msg.message +
                        "</div>" +
                        "</div>" +
                        "</div>";
                    
                    container.appendChild(messageDiv);
                    AdminMiniChat.lastMessageId = Math.max(AdminMiniChat.lastMessageId, parseInt(msg.id_message));
                });
                
                container.scrollTop = container.scrollHeight;
            },
            
            deleteMessage: function(messageId) {
                if (!confirm("Are you sure you want to delete this message?")) {
                    return;
                }
                
                var formData = new FormData();
                formData.append("action", "deleteMessage");
                formData.append("message_id", messageId);
                formData.append("id_employee", this.employeeId);
                formData.append("admin_token", this.adminToken);
                
                fetch(this.url, {
                    method: "POST",
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Supprimer visuellement le message
                        var msgElement = document.querySelector("[data-admin-msg-id=\"" + messageId + "\"]");
                        if (msgElement) {
                            msgElement.style.transition = "opacity 0.3s";
                            msgElement.style.opacity = "0";
                            setTimeout(function() {
                                msgElement.remove();
                            }, 300);
                        }
                    } else {
                        alert("Error: " + (data.error || "Failed to delete message"));
                    }
                })
                .catch(error => {
                    console.error("Error deleting message:", error);
                    alert("Error deleting message");
                });
            },
            
            send: function() {
                var input = document.getElementById("admin-chat-input");
                if (!input.value.trim()) return;
                
                var formData = new FormData();
                formData.append("action", "sendAdminMessage");
                formData.append("message", input.value.trim());
                formData.append("id_employee", this.employeeId);
                formData.append("admin_token", this.adminToken);
                
                fetch(this.url, {
                    method: "POST",
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        input.value = "";
                        this.loadMessages();
                    }
                });
            },
            
            startPolling: function() {
                this.timer = setInterval(() => {
                    this.loadMessages();
                }, 3000);
            }
        };
        
        document.addEventListener("DOMContentLoaded", function() {
            AdminMiniChat.init();
        });
        </script>
        ';
        
        return $output . $html;
    }

    /**
     * Formulaire de configuration
     */
    protected function renderForm()
    {
        $fields_form = [
            'form' => [
                'legend' => [
                    'title' => $this->l('Mini Chat Configuration'),
                    'icon' => 'icon-cogs',
                ],
                'input' => [
                    [
                        'type' => 'switch',
                        'label' => $this->l('Enable Mini Chat'),
                        'name' => 'MINICHAT_ENABLED',
                        'is_bool' => true,
                        'desc' => $this->l('Enable or disable the chat widget'),
                        'values' => [
                            [
                                'id' => 'active_on',
                                'value' => true,
                                'label' => $this->l('Enabled')
                            ],
                            [
                                'id' => 'active_off',
                                'value' => false,
                                'label' => $this->l('Disabled')
                            ]
                        ],
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l('Max characters per message'),
                        'name' => 'MINICHAT_MAX_CHARS',
                        'size' => 20,
                        'required' => true,
                        'desc' => $this->l('Between 50 and 500 characters')
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l('Refresh interval (milliseconds)'),
                        'name' => 'MINICHAT_REFRESH_INTERVAL',
                        'size' => 20,
                        'required' => true,
                        'desc' => $this->l('Between 1000 and 60000 ms')
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l('Max messages displayed'),
                        'name' => 'MINICHAT_MAX_MESSAGES',
                        'size' => 20,
                        'required' => true,
                        'desc' => $this->l('Between 10 and 100 messages')
                    ],
                ],
                'submit' => [
                    'title' => $this->l('Save'),
                ],
            ],
        ];
        
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);
        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitMiniChat';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            .'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        
        $helper->tpl_vars = [
            'fields_value' => $this->getConfigFormValues(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        ];
        
        return $helper->generateForm([$fields_form]);
    }

    /**
     * Récupération des valeurs de configuration
     */
    protected function getConfigFormValues()
    {
        return [
            'MINICHAT_ENABLED' => Configuration::get('MINICHAT_ENABLED', 1),
            'MINICHAT_MAX_CHARS' => Configuration::get('MINICHAT_MAX_CHARS', 200),
            'MINICHAT_REFRESH_INTERVAL' => Configuration::get('MINICHAT_REFRESH_INTERVAL', 3000),
            'MINICHAT_MAX_MESSAGES' => Configuration::get('MINICHAT_MAX_MESSAGES', 50),
        ];
    }

    /**
     * Hook Header - Chargement CSS et JS
     */
    public function hookDisplayHeader()
    {
        // Vérifier si le chat est activé
        if (!Configuration::get('MINICHAT_ENABLED')) {
            return;
        }

        // Charger CSS et JS
        $this->context->controller->addCSS($this->_path.'views/css/minichat.css', 'all');
        $this->context->controller->addJS($this->_path.'views/js/minichat.js');
    }
    
    /**
     * Hook Back Office Header
     */
    public function hookDisplayBackOfficeHeader()
    {
        // Ne rien faire ici pour l'instant
        return '';
    }

    /**
     * Hook Footer - Affichage du widget
     */
    public function hookDisplayFooter()
    {
        // Vérifier si le chat est activé
        if (!Configuration::get('MINICHAT_ENABLED')) {
            return '';
        }
        
        // Variables pour le template
        $this->context->smarty->assign([
            'minichat_ajax_url' => $this->context->link->getModuleLink('minichat', 'api'),
            'minichat_is_logged' => $this->context->customer->isLogged(),
            'minichat_customer_id' => (int)($this->context->customer->id ?? 0),
            'minichat_customer_name' => $this->context->customer->firstname ?? '',
            'minichat_max_chars' => (int)Configuration::get('MINICHAT_MAX_CHARS', 200),
            'minichat_refresh' => (int)Configuration::get('MINICHAT_REFRESH_INTERVAL', 3000),
            'minichat_is_banned' => $this->isCustomerBanned((int)($this->context->customer->id ?? 0)),
            'link' => $this->context->link,
        ]);
        
        return $this->display(__FILE__, 'views/templates/hook/displayFooter.tpl');
    }

    /**
     * Vérifier si un client est banni
     */
    public function isCustomerBanned($id_customer)
    {
        if (!$id_customer) {
            return false;
        }
        
        $sql = 'SELECT is_banned 
                FROM `'._DB_PREFIX_.'minichat_moderation` 
                WHERE id_customer = '.(int)$id_customer;
        
        return (bool)Db::getInstance()->getValue($sql);
    }

    /**
     * Bannir un client (méthode utilitaire)
     */
    public function banCustomer($id_customer, $reason = '')
    {
        if (!$id_customer) {
            return false;
        }

        $sql = 'INSERT INTO `'._DB_PREFIX_.'minichat_moderation` 
                (id_customer, is_banned, ban_reason, date_add, date_upd) 
                VALUES ('.(int)$id_customer.', 1, "'.pSQL($reason).'", NOW(), NOW())
                ON DUPLICATE KEY UPDATE 
                is_banned = 1, ban_reason = "'.pSQL($reason).'", date_upd = NOW()';
        
        return Db::getInstance()->execute($sql);
    }

    /**
     * Débannir un client (méthode utilitaire)
     */
    public function unbanCustomer($id_customer)
    {
        if (!$id_customer) {
            return false;
        }

        $sql = 'UPDATE `'._DB_PREFIX_.'minichat_moderation` 
                SET is_banned = 0, ban_reason = NULL, date_upd = NOW()
                WHERE id_customer = '.(int)$id_customer;
        
        return Db::getInstance()->execute($sql);
    }
    
    /**
     * Exporter le chat en fichier TXT
     */
    private function exportChatToTxt()
    {
        // Récupérer tous les messages
        $sql = 'SELECT m.*, 
                CASE 
                    WHEN m.id_employee > 0 THEN CONCAT("Admin ", e.firstname, " ", e.lastname)
                    WHEN c.firstname IS NOT NULL THEN CONCAT(c.firstname, " ", c.lastname)
                    ELSE "Guest"
                END as full_name,
                CASE 
                    WHEN m.id_employee > 0 THEN "ADMIN"
                    ELSE CONCAT("#", LPAD(m.id_customer, 4, "0"))
                END as user_id
                FROM `'._DB_PREFIX_.'minichat_messages` m
                LEFT JOIN `'._DB_PREFIX_.'customer` c ON m.id_customer = c.id_customer
                LEFT JOIN `'._DB_PREFIX_.'employee` e ON m.id_employee = e.id_employee
                WHERE m.is_deleted = 0
                ORDER BY m.date_add ASC';
        
        $messages = Db::getInstance()->executeS($sql);
        
        // Créer le contenu du fichier
        $content = "=================================\n";
        $content .= "MINICHAT EXPORT - " . date('Y-m-d H:i:s') . "\n";
        $content .= "Total Messages: " . count($messages) . "\n";
        $content .= "=================================\n\n";
        
        $currentDate = '';
        foreach ($messages as $msg) {
            $msgDate = date('Y-m-d', strtotime($msg['date_add']));
            
            // Ajouter un séparateur de date si nouveau jour
            if ($msgDate != $currentDate) {
                $currentDate = $msgDate;
                $content .= "\n--- " . date('l, F j, Y', strtotime($msg['date_add'])) . " ---\n\n";
            }
            
            // Format: [HH:MM] Nom (ID): Message
            $content .= "[" . date('H:i', strtotime($msg['date_add'])) . "] ";
            $content .= $msg['full_name'] . " (" . $msg['user_id'] . "): ";
            $content .= $msg['message'] . "\n";
        }
        
        $content .= "\n=================================\n";
        $content .= "END OF EXPORT\n";
        $content .= "=================================\n";
        
        // Envoyer le fichier
        header('Content-Type: text/plain; charset=utf-8');
        header('Content-Disposition: attachment; filename="minichat_export_' . date('Y-m-d_His') . '.txt"');
        header('Content-Length: ' . strlen($content));
        echo $content;
        exit;
    }
    
    /**
     * Réinitialiser le chat (effacer tous les messages)
     */
    private function resetChat()
    {
        // Option 1: Supprimer définitivement tous les messages
        // $sql = 'DELETE FROM `'._DB_PREFIX_.'minichat_messages`';
        
        // Option 2: Marquer tous les messages comme supprimés (soft delete)
        $sql = 'UPDATE `'._DB_PREFIX_.'minichat_messages` SET is_deleted = 1 WHERE is_deleted = 0';
        
        $result = Db::getInstance()->execute($sql);
        
        // Optionnel: Réinitialiser l'auto-increment (si suppression définitive)
        // Db::getInstance()->execute('ALTER TABLE `'._DB_PREFIX_.'minichat_messages` AUTO_INCREMENT = 1');
        
        return $result;
    }
}